<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\MiscBundle;

use Cyber\MiscBundle\SpreadsheetDataProcessor;
use Cyber\MiscBundle\SpreadsheetIterator;
use PHPUnit\Framework\TestCase;

/**
 * @covers \Cyber\MiscBundle\SpreadsheetIterator
 */
class SpreadsheetIteratorTest extends TestCase
{
    /**
     * @expectedException \RuntimeException
     * @expectedExceptionMessage box/spout dependency was not found.
     */
    public function testThrowOnMissingClass(): void
    {
        $iterator = new SpreadsheetIterator(__DIR__ . '/data/sheet-iterator-sample.csv');
        $iterator->throwOnMissingClass('App\Some\Super\Fake\Missing\Class');
    }

    public function testUninitializedIterator(): void
    {
        $iterator = new SpreadsheetIterator(__DIR__ . '/data/sheet-iterator-sample.csv');
        $this->assertNull($iterator->current());
        $this->assertNull($iterator->key());
        $this->assertFalse($iterator->valid());
        $iterator->next();
    }

    public function testIterator(): void
    {
        $expected = [
            ['id', 'name'],
            [1, 'smith'],
            [2, 'apple'],
            [3, 'zebra'],
        ];

        $expectedCount = \count($expected);

        $iterator = new SpreadsheetIterator(__DIR__ . '/data/sheet-iterator-sample.csv');
        $count    = 0;
        foreach ($iterator as $i => $row) {
            ++$count;
            $this->assertEquals($expected[$i], $row);
        }
        $this->assertEquals($expectedCount, $count);

        $count = 0;
        foreach ($iterator as $i => $row) {
            ++$count;
            $this->assertEquals($expected[$i], $row);
        }
        $this->assertEquals($expectedCount, $count);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter) not recognizing inheritdoc on dynamic classes
     */
    public function testIntegrationWithProcessor(): void
    {
        $iterator  = new SpreadsheetIterator(__DIR__ . '/data/sheet-iterator-sample.csv');
        $processor = new class() extends SpreadsheetDataProcessor {
            /**
             * {@inheritdoc}
             */
            protected function getColumnMapping(array $options): array
            {
                return [
                    'id'   => 'key',
                    'name' => 'secret',
                ];
            }
        };

        $result = \iterator_to_array($processor->process($iterator));

        $this->assertEquals([
            [
                'key'    => '1',
                'secret' => 'smith',
            ],
            [
                'key'    => '2',
                'secret' => 'apple',
            ],
            [
                'key'    => '3',
                'secret' => 'zebra',
            ],
        ], $result);
    }
}
