<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\DependencyInjection;

use Swift_Transport;
use Symfony\Component\Config\Definition\Builder\ArrayNodeDefinition;
use Symfony\Component\Config\Definition\Builder\NodeBuilder;
use Symfony\Component\Config\Definition\Builder\TreeBuilder;
use Symfony\Component\Config\Definition\ConfigurationInterface;

/**
 * This is the class that validates and merges configuration from your app/config files.
 *
 * To learn more see {@link http://symfony.com/doc/current/cookbook/bundles/configuration.html}
 */
class Configuration implements ConfigurationInterface
{
    // do not replace with class reference, as class may not exit
    const PAGINATOR_CLASS = '\Knp\Component\Pager\PaginatorInterface';

    /**
     * {@inheritdoc}
     */
    public function getConfigTreeBuilder(): TreeBuilder
    {
        $treeBuilder = new TreeBuilder('cyber_misc');

        /** @var ArrayNodeDefinition $rootNode */
        $rootNode = $treeBuilder->getRootNode();

        $this->addPaginationListenerConfigs($rootNode->children())
            //@formatter:off
            ->scalarNode('path_version_resolver_regex')
                ->defaultNull()
                ->info('If set will activate the patched FOS/Rest path version resolver. You need to provide regex for your version position in path.')
                ->example('/^\/api\/(?P<version>v?[0-9\.]+)\//')
            ->end()
            ->arrayNode('ses')
                ->info('Configuration for using AWS SES transport')
                ->children()
                    ->scalarNode('client')
                        ->info('Service id for the SesClient to use, if null, other configs must be provided.')
                    ->end()
                    ->scalarNode('key')
                        ->info('The key to use for authenticating with SES')
                    ->end()
                    ->scalarNode('secret')
                        ->info('The secret to use for authenticating with SES')
                    ->end()
                    ->scalarNode('region')
                        ->info('AWS region for sending email from. Leave null for default')
                    ->end()
                    ->scalarNode('version')
                        ->info('The version of AWS api to use. Leave null for default')
                    ->end()
                ->end()
                ->validate()
                    ->ifTrue(function ($nodeValue) {
                        return !\interface_exists(Swift_Transport::class);
                    })
                    ->thenInvalid(
                        \sprintf(
                            'Class "%s" does not exist. Please add "swiftmailer/swiftmailer" to use SES transport ' .
                            'provided by this bundle. You can remove "ses" key from configuration to disable it',
                            Swift_Transport::class
                        )
                    )
                ->end()
                ->validate()
                    ->ifTrue(function ($nodeValue) {
                        $hasClient = isset($nodeValue['client']);
                        $hasConfig = isset($nodeValue['key'], $nodeValue['secret']);

                        // if none are configured
                        return !$hasClient && !$hasConfig;
                    })
                    ->thenInvalid('Either client service or key + secret need to be specified')
                ->end()
                ->validate()
                    ->ifTrue(function ($nodeValue) {
                        $hasClient = isset($nodeValue['client']);
                        $hasConfig = isset($nodeValue['key']) || isset($nodeValue['secret']);

                        // if both are configured, probably an error
                        return $hasClient && $hasConfig;
                    })
                    ->thenInvalid(
                        'Both client service and key + secret are configured. '
                        . 'Key + Secret would be ignored so we consider this a config error.'
                    )
                ->end()
            ->end()
        ->end();
        //@formatter:on

        return $treeBuilder;
    }

    private function addPaginationListenerConfigs(NodeBuilder $node): NodeBuilder
    {
        //@formatter:off
        $node->arrayNode('paginator_response_listener')
            ->info('If enable you can return QueryBuilder or Query from controller and it will be auto paginated.')
            ->canBeEnabled()
            ->validate()
                ->ifTrue(function ($nodeValue) {
                    return $nodeValue['enabled'] && !\interface_exists(self::PAGINATOR_CLASS);
                })
                ->thenInvalid(
                    \sprintf(
                        'Class "%s" does not exist. Please add "knplabs/knp-paginator-bundle" to use paginator ' .
                        'response listener. Set this value to false to disable this listener.',
                        self::PAGINATOR_CLASS
                    )
                )
            ->end()
            ->children()
                ->scalarNode('default_page')
                    ->info('The page to show when one is not specified')
                    ->defaultValue(1)
                ->end()
                ->scalarNode('default_page_size')
                    ->info('Default items per page')
                    ->defaultValue(25)
                ->end()
                ->scalarNode('min_page_size')
                    ->info('Minimum allowed items per page')
                    ->defaultValue(10)
                ->end()
                ->scalarNode('max_page_size')
                    ->info('Maximum allowed items per page')
                    ->defaultValue(1000)
                ->end()
            ->end()
        ->end();
        //@formatter:on

        return $node;
    }
}
