<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\DependencyInjection;

use Aws\Ses\SesClient;
use Cyber\MiscBundle\Event\Subscriber\PaginationSubscriber;
use Cyber\MiscBundle\Service\AwsSesMailerTransport;
use Cyber\MiscBundle\Service\FosRestPathResolver;
use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Loader;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

/**
 * This is the class that loads and manages your bundle configuration.
 *
 * @see http://symfony.com/doc/current/cookbook/bundles/extension.html
 */
class CyberMiscExtension extends Extension
{
    /**
     * {@inheritdoc}
     */
    public function load(array $configs, ContainerBuilder $container): void
    {
        $loader = new Loader\YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/config'));
        $loader->load('services.yaml');

        $configuration = new Configuration();
        $config        = $this->processConfiguration($configuration, $configs);

        if ($config['paginator_response_listener']['enabled']) {
            $container->register(PaginationSubscriber::class)
                ->setAutoconfigured(true)
                ->setAutowired(true)
                ->setArgument('$defaults', $config['paginator_response_listener']);
        }

        if (isset($config['ses'])) {
            $this->configureSesTransport($config['ses'], $container);
        }

        if (isset($config['path_version_resolver_regex'])) {
            $this->configureRestPathResolver($config['path_version_resolver_regex'], $container);
        }
    }

    /**
     * @param array<mixed>     $ses
     * @param ContainerBuilder $container
     */
    private function configureSesTransport($ses, ContainerBuilder $container): void
    {
        $serviceId = AwsSesMailerTransport::class;
        $def       = $container->register($serviceId);
        $container->setAlias(\sprintf('swiftmailer.mailer.transport.%s', $serviceId), $serviceId);

        if (isset($ses['client'])) {
            // if providing client inject that as the argument directly
            $def->addArgument(new Reference($ses['client']));

            return;
        }

        if (!\class_exists(SesClient::class)) {
            throw new InvalidConfigurationException(
                \sprintf(
                    'Class "%s" not found. Either include "%s" or provide SES client service via "%s" configuration key',
                    SesClient::class,
                    'aws/aws-sdk-php',
                    'cyber_misc.ses.client'
                )
            );
        }

        // if not providing client service configure the parameters for our local client definition
        $def->addArgument(new Reference('cyber.misc.ses_client'));
        foreach ($ses as $key => $value) {
            $container->setParameter('cyber.misc.ses.' . $key, $value);
        }
    }

    private function configureRestPathResolver(string $pathRegex, ContainerBuilder $container): void
    {
        $serviceId = FosRestPathResolver::class;
        $def       = $container->register($serviceId);
        $def->setArgument(0, $pathRegex);
    }
}
