<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\DependencyInjection;

use Symfony\Component\Config\Definition\Builder\ArrayNodeDefinition;
use Symfony\Component\Config\Definition\Builder\NodeBuilder;
use Symfony\Component\Config\Definition\Builder\TreeBuilder;
use Symfony\Component\Config\Definition\ConfigurationInterface;

/**
 * This is the class that validates and merges configuration from your app/config files.
 *
 * To learn more see {@link http://symfony.com/doc/current/cookbook/bundles/configuration.html}
 */
class Configuration implements ConfigurationInterface
{
    // do not replace with class reference, as class may not exit
    const PAGINATOR_CLASS = '\Knp\Component\Pager\PaginatorInterface';

    /**
     * @inheritdoc
     */
    public function getConfigTreeBuilder(): TreeBuilder
    {
        $treeBuilder = new TreeBuilder('cyber_misc');

        /** @var ArrayNodeDefinition $rootNode */
        $rootNode = $treeBuilder->getRootNode();

        $this->addPaginationListenerConfigs($rootNode->children())
            //@formatter:off
            ->scalarNode('path_version_resolver_regex')
                ->defaultNull()
                ->info('If set will activate the patched FOS/Rest path version resolver. You need to provide regex for your version position in path.')
                ->example('/^\/api\/(?P<version>v?[0-9\.]+)\//')
            ->end();
        //@formatter:on

        return $treeBuilder;
    }

    private function addPaginationListenerConfigs(NodeBuilder $node): NodeBuilder
    {
        //@formatter:off
        $node->arrayNode('paginator_response_listener')
            ->info('If enable you can return QueryBuilder or Query from controller and it will be auto paginated.')
            ->canBeEnabled()
            ->validate()
                ->ifTrue(function ($nodeValue) {
                    return $nodeValue['enabled'] && !\interface_exists(self::PAGINATOR_CLASS);
                })
                ->thenInvalid(
                    \sprintf(
                        'Class "%s" does not exist. Please add "knplabs/knp-paginator-bundle" to use paginator ' .
                        'response listener. Set this value to false to disable this listener.',
                        self::PAGINATOR_CLASS
                    )
                )
            ->end()
            ->children()
                ->scalarNode('default_page')
                    ->info('The page to show when one is not specified')
                    ->defaultValue(1)
                ->end()
                ->scalarNode('default_page_size')
                    ->info('Default items per page')
                    ->defaultValue(25)
                ->end()
                ->scalarNode('min_page_size')
                    ->info('Minimum allowed items per page')
                    ->defaultValue(10)
                ->end()
                ->scalarNode('max_page_size')
                    ->info('Maximum allowed items per page')
                    ->defaultValue(1000)
                ->end()
            ->end()
        ->end();
        //@formatter:on

        return $node;
    }
}
