### Cyber\MiscBundle\SpreadsheetDataProcessor
A utility class to simplify your spreadsheet/table type data processing. 

## Features
* Allows data columns to be in any order
* Can tolerate extra or missing data
* Changes in column name do not result in major refactoring.
* Low memory footprint thanks to iterators/generators

## Requirements
* First row of data must be column headings
* Rows must be simple 0-index based arrays

## Usage

Start by extending the processor class:

```php
<?php
use \Cyber\MiscBundle\SpreadsheetDataProcessor;

class YourProcessor extends SpreadsheetDataProcessor
{
    protected function getColumnMapping(array $options): array
    {
        // provide mapping to your columns
        return [
            'col 1'     => 'id',
            'other col' => 'data',
        ];
    }

    protected function mapRow(array &$data, int $line, array $options)
    {
        // convert to your object
        return new YourObject($data['id'], $data['data']);
    }
}
```

Next, pass your data through the processor:

```php
<?php

$yourDataIterator = getMyDataIterator();

$processedIterator = (new YourYourProcessor())->process($yourDataIterator);

foreach($processedIterator as $yourObject){
    // do something with your data
}
```

## Options
As a second parameter to `process()` you can pass options array. This array is then passed to `mapRow()` and 
`getColumnMapping()` functions. You can pass arbitrary options there to customize the return values or processing
withing those functions. 

> For example, maybe you have multiple versions of the data file with slight variations that need to go
> through the same *Processor*. In this case you could pass the version via options and perform necessary
> handling in each of the functions.

There are 2 pre-defined options that affect the behavior of `process()` function:

* SpreadsheetDataProcessor::OPT_IGNORE_EXTRA_COLUMNS - if set to true, will cause process to silently ignore extra data
in any particular row.
* SpreadsheetDataProcessor::OPT_IGNORE_MISSING_COLUMNS - if set to true, will cause insufficient columns to be fill with 
empty strings.

### OPT_IGNORE_EXTRA_COLUMNS
Sometimes (especially in excel sheets) user may tab too far in and create an extra value in just a single row like so:

| col 1 | col 2 | (blank) |
| ----- | ----- | ------- |
| A     | B     | (blank) |
| C     | D     | some value (usually space which remains unnoticed) |
| E     | F     | (blank) |

> the `(blank)` values means they are not there at all (not that they are empty). In the above example
> sheet reader would return `['col 1', 'col 2']` as the first array of data while returning  `['C', 'D', 'some value']`
> as the third

In such case if `OPT_IGNORE_EXTRA_COLUMNS` is not set an exception will be throw during processing of line 3 of the
data since there are more data elements than number of defined columns.

If the option is set however, the extra elements will be silently ignored.

### OPT_IGNORE_MISSING_COLUMNS

This is kind of the opposite of extra columns. The sheet might look like this:

| col 1 | col 2   |
| ----- | ------- |
| A     | B       |
| C     | (blank) |
| E     | F       |

> In this case each iteration will have 2 elements in array except for 3rd which will have just one

In such case if `OPT_IGNORE_MISSING_COLUMNS` is not set an exception will be throw during processing of line 3 of the
 data since there are less data elements than number of defined columns.
 
If the option is set however, the missing data will be filled with empty string values.
