<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\DependencyInjection;

use Symfony\Component\Config\Definition\Builder\ArrayNodeDefinition;
use Symfony\Component\Config\Definition\Builder\NodeBuilder;
use Symfony\Component\Config\Definition\Builder\TreeBuilder;
use Symfony\Component\Config\Definition\ConfigurationInterface;

/**
 * This is the class that validates and merges configuration from your app/config files.
 *
 * To learn more see {@link http://symfony.com/doc/current/cookbook/bundles/configuration.html}
 */
class Configuration implements ConfigurationInterface
{
    // do not replace with class reference, as class may not exit
    const PAGINATOR_CLASS = '\Knp\Component\Pager\PaginatorInterface';

    /**
     * @inheritdoc
     */
    public function getConfigTreeBuilder(): TreeBuilder
    {
        $treeBuilder = new TreeBuilder('cyber_misc');

        /** @var ArrayNodeDefinition $rootNode */
        $rootNode = $treeBuilder->getRootNode();

        $this->addPaginationListenerConfigs($rootNode->children())
            //@formatter:off
            ->scalarNode('path_version_resolver_regex')
                ->defaultNull()
                ->info('If set will activate the patched FOS/Rest path version resolver. You need to provide regex for your version position in path.')
                ->example('/^\/api\/(?P<version>v?[0-9\.]+)\//')
            ->end()
            ->arrayNode('kafka')
                ->canBeEnabled()
                ->validate()
                    ->ifTrue(function ($configs) {
                        return \count($configs) && !\class_exists('\\RdKafka\\Conf', false);
                    })
                    ->thenInvalid('Cannot find RdKafka\\Conf class ensure that ext-rdkafka extension is active.')
                ->end()
                ->children()
                    ->arrayNode('configs')->info('List of broker configurations, used as config for topics')
                        ->prototype('array')
                            ->children()
                                ->scalarNode('group')
                                    ->info('All consumer with the same group.id will consume different partitions.')
                                    ->isRequired()
                                    ->cannotBeEmpty()
                                ->end()
                                ->scalarNode('brokers')
                                    ->info('Comma delimited list of brokers host:port')
                                    ->isRequired()
                                    ->cannotBeEmpty()
                                ->end()
                                ->enumNode('default_offset')->values(['earliest', 'latest', 'error'])->isRequired()->end()
                                ->booleanNode('auto_offset_store')
                                    ->info('Enable automatic reporting of consumed messages. Must be off when used for transport.')
                                    ->isRequired()
                                ->end()
                                ->booleanNode('emit_eof')
                                    ->info('Enable extra EndOfFile event for producers, else they simply block until timeout upon reaching end of messages.')
                                    ->defaultTrue()
                                ->end()
                                ->enumNode('security_proto')->values(['plaintext', 'ssl', 'sasl_plaintext', 'sasl_ssl'])
                                    ->info('security protocol for brokers')
                                    ->defaultValue('ssl')
                                ->end()
                                ->arrayNode('extra')
                                    ->info('Additional configs to pass to kafka; see: https://github.com/confluentinc/librdkafka/blob/master/CONFIGURATION.md')
                                    ->prototype('scalar')
                                    ->end()
                                ->end()
                            ->end()
                        ->end()
                    ->end()
                    ->arrayNode('topics')
                        ->info('List of topics to create consumers/producers for. Array keys is used as reference name.')
                        ->arrayPrototype()
                            ->children()
                                ->scalarNode('config')
                                    ->info('Name of the config from configs list to use for constructing consumer/producer pair for this topic.')
                                    ->isRequired()
                                    ->cannotBeEmpty()
                                ->end()
                                ->scalarNode('topic_name')
                                    ->info('Name of the kafka topic consumers/producers will read/write to.')
                                    ->isRequired()
                                    ->cannotBeEmpty()
                                ->end()
                                ->booleanNode('lazy')->info('Mark services as lazy')->defaultFalse()->end()
                                ->booleanNode('transport')->info('Register topic with transport')->defaultFalse()->end()
                                ->booleanNode('streamer')->info('Register topic as a streamer')->defaultFalse()->end()
                            ->end()
                        ->end()
                    ->end()
                ->end()
            ->end();
        //@formatter:on

        return $treeBuilder;
    }

    private function addPaginationListenerConfigs(NodeBuilder $node): NodeBuilder
    {
        //@formatter:off
        $node->arrayNode('paginator_response_listener')
            ->info('If enable you can return QueryBuilder or Query from controller and it will be auto paginated.')
            ->canBeEnabled()
            ->validate()
                ->ifTrue(function ($nodeValue) {
                    return $nodeValue['enabled'] && !\interface_exists(self::PAGINATOR_CLASS);
                })
                ->thenInvalid(
                    \sprintf(
                        'Class "%s" does not exist. Please add "knplabs/knp-paginator-bundle" to use paginator ' .
                        'response listener. Set this value to false to disable this listener.',
                        self::PAGINATOR_CLASS
                    )
                )
            ->end()
            ->children()
                ->scalarNode('default_page')
                    ->info('The page to show when one is not specified')
                    ->defaultValue(1)
                ->end()
                ->scalarNode('default_page_size')
                    ->info('Default items per page')
                    ->defaultValue(25)
                ->end()
                ->scalarNode('min_page_size')
                    ->info('Minimum allowed items per page')
                    ->defaultValue(10)
                ->end()
                ->scalarNode('max_page_size')
                    ->info('Maximum allowed items per page')
                    ->defaultValue(1000)
                ->end()
            ->end()
        ->end();
        //@formatter:on

        return $node;
    }
}
