<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\DependencyInjection;

use Cyber\MiscBundle\Event\Subscriber\PaginationSubscriber;
use Cyber\MiscBundle\Kafka\KafkaCallbacks;
use Cyber\MiscBundle\Kafka\KafkaStreamerInterface;
use Cyber\MiscBundle\Kafka\KafkaTransportFactory;
use Cyber\MiscBundle\Kafka\SingleTopicConsumer;
use Cyber\MiscBundle\Kafka\SingleTopicProducer;
use Cyber\MiscBundle\Service\FosRestPathResolver;
use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\Argument\ServiceClosureArgument;
use Symfony\Component\DependencyInjection\ChildDefinition;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Loader;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

/**
 * This is the class that loads and manages your bundle configuration.
 *
 * @see http://symfony.com/doc/current/cookbook/bundles/extension.html
 *
 * @phpstan-type BrokerConf array{group: string, group: string, brokers: string,
 *     security_proto: null|string, default_offset: string, emit_eof: bool, auto_offset_store: bool,
 *     extra: array<string, scalar>
 *         }
 * @phpstan-type TopicConf array{config: string, topic_name: string, transport: bool, streamer: bool, lazy: bool}
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CyberMiscExtension extends Extension
{
    /**
     * @inheritdoc
     */
    public function load(array $configs, ContainerBuilder $container): void
    {
        $loader = new Loader\YamlFileLoader($container, new FileLocator(__DIR__ . '/../../config'));
        $loader->load('services.yaml');

        $configuration = new Configuration();
        $config        = $this->processConfiguration($configuration, $configs);

        if ($config['paginator_response_listener']['enabled']) {
            $container->register(PaginationSubscriber::class)
                ->setAutoconfigured(true)
                ->setAutowired(true)
                ->setArgument('$defaults', $config['paginator_response_listener']);
        }

        if (isset($config['path_version_resolver_regex'])) {
            $this->configureRestPathResolver($config['path_version_resolver_regex'], $container);
        }

        if (isset($config['kafka'])) {
            $this->provideKafkaStreamers($container, $config['kafka']);
        }
    }

    private function configureRestPathResolver(string $pathRegex, ContainerBuilder $container): void
    {
        $serviceId = FosRestPathResolver::class;
        $def       = $container->register($serviceId);
        $def->setArgument(0, $pathRegex);
    }

    /**
     * @param array{enabled: bool, configs: array<string, BrokerConf>, topics: array<string, TopicConf>} $kafkaConfigs
     */
    private function provideKafkaStreamers(ContainerBuilder $container, array $kafkaConfigs): void
    {
        if (!$kafkaConfigs['enabled']) {
            return;
        }

        $this->validateTopicConfigs($kafkaConfigs);

        $configIds        = $this->registerKafkaConfigServices($container, $kafkaConfigs['configs']);
        $transportFactory = $container->getDefinition(KafkaTransportFactory::class);

        foreach ($kafkaConfigs['topics'] as $topicRef => $topicConf) {
            [$configId, $prodConfId] = $configIds[$topicConf['config']];

            $topicName  = $topicConf['topic_name'];
            $consumerId = \sprintf('cyber.misc.%s_kafka.consumer', $topicRef);
            $producerId = \sprintf('cyber.misc.%s_kafka.producer', $topicRef);
            $consumer   = $container->setDefinition($consumerId, new ChildDefinition('cyber.misc.kafka.consumer'));
            $consumer->setArgument(0, new Reference($configId))
                ->addMethodCall('assignTopic', [$topicName])
                ->addMethodCall('subscribe', [[$topicName]])
                ->setLazy($topicConf['lazy']);

            $producer = $container->setDefinition($producerId, new ChildDefinition('cyber.misc.kafka.producer'));
            $producer->setArgument(0, new Reference($prodConfId))
                ->addMethodCall('assignTopic', [$topicName])
                ->setLazy($topicConf['lazy']);

            if ($topicConf['transport']) {
                $transportFactory->addMethodCall('registerTopic', [
                    $topicRef,
                    new ServiceClosureArgument(new Reference($consumerId)),
                    new ServiceClosureArgument(new Reference($producerId)),
                ]);
            }

            if ($topicConf['streamer']) {
                $streamerId = \sprintf('cyber.misc.kafka.%s_streamer', $topicRef);
                $streamer   = $container->setDefinition($streamerId, new ChildDefinition('cyber.misc.kafka.streamer'));
                $streamer->setArgument('$consumerClosure', new ServiceClosureArgument(new Reference($consumerId)));
                $variableName = $this->toCamelCase($topicRef) . 'Streamer';
                $container->setAlias(KafkaStreamerInterface::class . ' $' . $variableName, $streamerId);
            }

            $variableName = $this->toCamelCase($topicRef) . 'Consumer';
            // set alias by variable name
            $container->setAlias(SingleTopicConsumer::class . ' $' . $variableName, $consumerId);
            $container->setAlias('\RdKafka\KafkaConsumer $' . $variableName, $consumerId);

            $variableName = $this->toCamelCase($topicRef) . 'Producer';
            // set alias by variable name
            $container->setAlias(SingleTopicProducer::class . ' $' . $variableName, $producerId);
            $container->setAlias('\RdKafka\Producer $' . $variableName, $producerId);
        }
    }

    /**
     * @param array{enabled: bool, configs: array<string, BrokerConf>, topics: array<string, TopicConf>} $kafkaConfigs
     */
    private function validateTopicConfigs($kafkaConfigs): void
    {
        foreach ($kafkaConfigs['topics'] as $topicRef => $topicConf) {
            $configName = $topicConf['config'];

            if ($topicConf['transport'] && $kafkaConfigs['configs'][$configName]['auto_offset_store']) {
                throw new InvalidConfigurationException(
                    'When using kafka topic for transport, the "auto_offset_store", must be disabled. Topic: ' . $topicRef
                );
            }

            if ($topicConf['streamer'] && $kafkaConfigs['configs'][$configName]['auto_offset_store']) {
                throw new InvalidConfigurationException(
                    'When using kafka topic for streamer, the "auto_offset_store", must be disabled. Topic: ' . $topicRef
                );
            }

            if (!isset($kafkaConfigs['configs'][$configName])) {
                throw new InvalidConfigurationException(
                    \sprintf('Broker config with name "%s" is not found for topic: %s', $configName, $topicRef)
                );
            }
        }
    }

    private function toCamelCase(string $name): string
    {
        return \lcfirst(\preg_replace_callback('/(^|_|\.)+(.)/', function ($match) {
            return ('.' === $match[1] ? '_' : '') . \mb_strtoupper($match[2]);
        }, $name) ?? '');
    }

    /**
     * @return array<string, array{0: string, 1: string}>
     */
    private function registerKafkaConfigServices(ContainerBuilder $container, array $configs): array
    {
        $configIds = [];

        foreach ($configs as $name => $brokerConf) {
            $consumerId       = \sprintf('cyber.misc.%s_kafka.configuration', $name);
            $producerId       = \sprintf('cyber.misc.%s_kafka.producer_conf', $name);
            $configIds[$name] = [$consumerId, $producerId];
            $rdConf           = $container->setDefinition($consumerId, new ChildDefinition('cyber.misc.kafka.config'));
            $prodConf         = $container->setDefinition($producerId, new ChildDefinition('cyber.misc.kafka.config'));

            $meta                             = $brokerConf['extra'];
            $meta['group.id']                 = $brokerConf['group'];
            $meta['metadata.broker.list']     = $brokerConf['brokers'];
            $meta['security.protocol']        = $brokerConf['security_proto'] ?? 'plaintext';
            $meta['auto.offset.reset']        = $brokerConf['default_offset'];
            $meta['enable.partition.eof']     = $brokerConf['emit_eof'] ? 'true' : 'false';
            $meta['enable.auto.offset.store'] = $brokerConf['auto_offset_store'] ? 'true' : 'false';

            $consumerOnly = [
                'group.id'                 => true,
                'enable.auto.offset.store' => true,
                'enable.partition.eof'     => true,
                'auto.offset.reset'        => true,
            ];

            foreach ($meta as $key => $value) {
                $rdConf->addMethodCall('set', [$key, (string) $value]);
                if (!isset($consumerOnly[$key])) {
                    $prodConf->addmethodCall('set', [$key, (string) $value]);
                }
            }

            $rdConf->addMethodCall('setRebalanceCb', [[new Reference(KafkaCallbacks::class), 'rebalance']]);
        }

        return $configIds;
    }
}
