<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package.
 */

declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\Kafka;

use Closure;
use Generator;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use RdKafka\Message;

class KafkaStreamer implements KafkaStreamerInterface
{
    use KafkaConsumerHelperTrait;

    private bool $stopRequested = false;

    private int $idleTimeSec = 0;

    /** @var callable[] */
    private array $idleCallbacks = [];

    private Closure $consumerClosure;

    public function __construct(Closure $consumerClosure, LoggerInterface $logger = null)
    {
        $this->consumerClosure = $consumerClosure;
        $this->logger          = $logger ?: new NullLogger();
    }

    /**
     * @param int $idleTimeSec if no messages received for this number of seconds, idle callbacks will be triggered
     *
     * @throws \RdKafka\Exception
     *
     * @return Generator<Message>
     */
    public function streamEvents(int $idleTimeSec = 0): Generator
    {
        $this->stopRequested = false;
        $this->idleTimeSec   = $idleTimeSec;

        $idleStart = \time();
        $idleSent  = false;
        /** @var SingleTopicConsumer $consumer */
        $consumer = ($this->consumerClosure)();

        $this->logger->debug('Starting to stream events', [
            'topic' => $consumer->topicInfo(), 'idleTimeout' => $idleTimeSec,
        ]);

        foreach ($this->streamRawEvents($consumer) as $message) {
            if (null !== $message) {
                // only yield messages that are not idle and reset the timer
                $idleStart = \time();
                $idleSent  = false;
                yield $message;

                continue;
            }

            if (!$idleSent && (\time() - $idleStart) > $this->idleTimeSec) {
                $idleSent = true;
                $this->dispatchIdleTimeout();
            }
        }
    }

    /**
     * Calling this will cause the streaming event generator to stop after current iteration.
     *
     * The consumer will be closed.
     */
    public function stop(): void
    {
        $this->stopRequested = true;
    }

    public function addIdleCallback(callable $callback): void
    {
        $this->idleCallbacks[] = $callback;
    }

    /**
     * @throws \RdKafka\Exception
     *
     * @return Generator<null|Message>
     */
    private function streamRawEvents(SingleTopicConsumer $consumer): Generator
    {
        while (!$this->stopRequested) {
            $message  = $consumer->consume(250);
            $message  = $this->preProcessMessage($message);
            if (null === $message) {
                // messages is sometimes null at the initialization stages, just ignore it.
                yield null;
                continue;
            }

            yield $message;
            $consumer->storeOffset($message->partition, $message->offset); // record offset after message processed
        }
    }

    private function dispatchIdleTimeout(): void
    {
        $this->logger->debug('Dispatching idle callbacks');
        foreach ($this->idleCallbacks as $callback) {
            $callback($this);
        }
    }
}
