<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\Kafka;

use LogicException;
use Psr\Log\LoggerInterface;
use RdKafka\Exception;
use Symfony\Component\Messenger\Envelope;
use Symfony\Component\Messenger\Stamp\TransportMessageIdStamp;
use Symfony\Component\Messenger\Transport\Serialization\SerializerInterface;
use Symfony\Component\Messenger\Transport\TransportInterface;

class KafkaTransport implements TransportInterface
{
    use KafkaConsumerHelperTrait;

    private bool $producedOnce = false;

    private SerializerInterface $serializer;

    private SingleTopicConsumer $consumer;

    private SingleTopicProducer $producer;

    public function __construct(
        SingleTopicConsumer $consumer,
        SingleTopicProducer $producer,
        SerializerInterface $serializer,
        LoggerInterface $logger
    ) {
        $this->consumer   = $consumer;
        $this->producer   = $producer;
        $this->serializer = $serializer;
        $this->logger     = $logger;
    }

    public function __destruct()
    {
        if ($this->producedOnce) {
            // lots of async stuff so we must flush before quitting, else things could get lost
            $this->producer->flush(30 * 1000);
        }
    }

    public function get(): iterable
    {
        $message = $this->consumer->consume(10);
        $message = $this->preProcessMessage($message);
        if (null === $message) {
            return [];
        }

        $envelope = $this->serializer->decode([
            'body' => $message->payload,
        ]);

        return [
            $envelope->with(
                new TransportMessageIdStamp([$message->partition, $message->offset])
            ),
        ];
    }

    public function ack(Envelope $envelope): void
    {
        $stamp = $envelope->last(TransportMessageIdStamp::class);
        if (!$stamp instanceof TransportMessageIdStamp) {
            throw new LogicException('No TransportMessageIdStamp found on the Envelope.');
        }

        /** @var array{int, int} $id from get */
        $id = $stamp->getId();
        $this->consumer->storeOffset(...$id);
    }

    public function reject(Envelope $envelope): void
    {
        // no deletion for kafka, just ack so it does not come up again
        $this->ack($envelope);
    }

    public function send(Envelope $envelope): Envelope
    {
        $id             = \bin2hex(\random_bytes(16));
        $encodedMessage = $this->serializer->encode($envelope);
        for ($i = 0; $i < 5; ++$i) {
            try {
                $this->producer->produce($encodedMessage['body'], $id);
                break;
            } catch (Exception $ex) {
                // unknown topic can happen during auto creation step, which is ok, give it a couple tries before failing.
                if (!\mb_strpos($ex->getMessage(), 'Unknown topic')) {
                    throw $ex;
                }
                $this->logger->warning('Unknown topic, error. Ok if topic will auto create. If you keep seeing the message check your configs');
                \sleep(1);
            }
        }
        $this->producedOnce = true;

        return $envelope->with(new TransportMessageIdStamp($id));
    }
}
