<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\Kafka;

use OutOfBoundsException;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use Symfony\Component\Messenger\Transport\Serialization\SerializerInterface;
use Symfony\Component\Messenger\Transport\TransportFactoryInterface;
use Symfony\Component\Messenger\Transport\TransportInterface;

/**
 * @implements TransportFactoryInterface<KafkaTransport>
 */
class KafkaTransportFactory implements TransportFactoryInterface
{
    /** @var array<string, array{0: \Closure, 1: \Closure}> */
    private $topics = [];

    /**
     * @var LoggerInterface
     */
    private $logger;

    public function __construct(LoggerInterface $logger = null)
    {
        $this->logger = $logger ?: new NullLogger();
    }

    /**
     * @inheritDoc
     *
     * @param mixed[] $options
     */
    public function createTransport(string $dsn, array $options, SerializerInterface $serializer): TransportInterface
    {
        $topic = \mb_substr($dsn, 8);
        if (!isset($this->topics[$topic])) {
            throw new OutOfBoundsException('Specified topic is not registered with transport. Did you set "transport: true" in the kafka config');
        }
        [$consumer, $producer] = $this->topics[$topic];

        return new KafkaTransport($consumer(), $producer(), $serializer, $this->logger);
    }

    /**
     * @inheritDoc
     *
     * @param mixed[] $options
     */
    public function supports(string $dsn, array $options): bool
    {
        return 0 === \mb_strpos($dsn, 'kafka://');
    }

    public function registerTopic(string $topic, \Closure $consumer, \Closure $producer): void
    {
        $this->topics[$topic] = [$consumer, $producer];
    }
}
