<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\Service;

use Aws\Credentials\Credentials;
use Aws\Exception\AwsException;
use Aws\Ses\SesClient;
use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Mailer\Exception\TransportException;
use Symfony\Component\Mailer\Header\MetadataHeader;
use Symfony\Component\Mailer\SentMessage;
use Symfony\Component\Mailer\Transport\AbstractTransport;
use Symfony\Component\Mime\Message;

class AwsSesMailerTransport extends AbstractTransport
{
    private SesClient $client;

    public function __construct(
        SesClient $client,
        EventDispatcherInterface $dispatcher = null,
        LoggerInterface $logger = null
    ) {
        $this->client = $client;
        parent::__construct($dispatcher, $logger);
    }

    protected function doSend(SentMessage $message): void
    {
        $request = [
            'Destination'      => [
                'ToAddresses' => $this->stringifyAddresses($message->getEnvelope()->getRecipients()),
            ],
            'ReplyToAddresses' => [$message->getEnvelope()->getSender()->toString()],
            'Source'           => $message->getEnvelope()->getSender()->toString(),
            'RawMessage'       => [
                'Data' => $message->toString(),
            ],
        ];

        $originalMsg = $message->getOriginalMessage();

        if ($originalMsg instanceof Message) {
            $headers         = $originalMsg->getHeaders();
            $configSetHeader = $headers->get('X-SES-CONFIGURATION-SET');
            if ($configSetHeader) {
                $request['ConfigurationSetName'] = $configSetHeader->getBodyAsString();
            }

            $sourceArnHeader = $headers->get('X-SES-SOURCE-ARN');
            if ($sourceArnHeader) {
                $request['SourceArn'] = $sourceArnHeader->getBodyAsString();
            }

            foreach ($headers->all() as $header) {
                if ($header instanceof MetadataHeader) {
                    $request['Tags'][] = ['Name' => $header->getKey(), 'Value' => $header->getValue()];
                }
            }
        }

        try {
            $result = $this->client->sendRawEmail($request);
            /** @var string $messageId */
            $messageId = $result['MessageId'];
            $message->setMessageId($messageId);
        } catch (AwsException $e) {
            throw new TransportException(
                \sprintf(
                    'Unable to send an email: %s (code %s).',
                    $e->getAwsErrorMessage() ?: $e->getMessage(),
                    $e->getAwsErrorCode() ?: $e->getCode()
                ),
                $e->getCode(),
                $e
            );
        }
    }

    public function __toString(): string
    {
        $endpoint = $this->client->getEndpoint();
        /** @var Credentials $creds */
        $creds = $this->client->getCredentials()->wait();

        return \sprintf('aws+sdk://%s@%s', $creds->getAccessKeyId(), $endpoint->getHost());
    }
}
