<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\Kafka;

use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use RdKafka\Conf;
use RdKafka\Producer;
use RdKafka\ProducerTopic;

class SingleTopicProducer extends Producer
{
    /** @var ProducerTopic */
    private $topic;

    /**
     * @var LoggerInterface
     */
    private $logger;

    private bool $firstMessage = true;

    /**
     * @param Conf $conf
     */
    public function __construct($conf = null, LoggerInterface $logger = null)
    {
        parent::__construct($conf);
        $this->logger = $logger ?? new NullLogger();
    }

    public function assignTopic(string $name): void
    {
        $this->topic = $this->newTopic($name);
    }

    /**
     * @throws \RdKafka\Exception
     */
    public function produceex(int $partition, int $msgflags, string $payload = null, string $key = null, string $opaque = null): void
    {
        $this->logger->debug('Producing message to topic.', [
            'topic'   => $this->topic->getName(),
            'payload' => $payload,
        ]);

        try {
            $this->topic->produce($partition, $msgflags, $payload, $key, $opaque);
        } catch(\RdKafka\Exception $ex) {
            if ($this->firstMessage && RD_KAFKA_RESP_ERR__UNKNOWN_TOPIC === $ex->getCode()) {
                // sometimes we might get topic errors, retry but only once.
                $this->firstMessage = false;
                \sleep(1);
                $this->produceex($partition, $msgflags, $payload, $key, $opaque);
            }
        }
        $this->poll(0);
    }

    /**
     * @throws \RdKafka\Exception
     */
    public function produce(string $payload = null, string $key = null, string $opaque = null): void
    {
        $this->produceex(RD_KAFKA_PARTITION_UA, 0, $payload, $key, $opaque);
    }
}
