<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\Kafka;

use Exception;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use RdKafka\KafkaConsumer;
use RdKafka\TopicPartition;

class KafkaCallbacks
{
    /** @var LoggerInterface */
    private LoggerInterface $logger;

    public function __construct(/* private */ LoggerInterface $logger = null)
    {
        $this->logger = $logger ?: new NullLogger();
    }

    /**
     * @param KafkaConsumer         $kafka
     * @param int                   $err
     * @param null|TopicPartition[] $partitions
     *
     * @throws \RdKafka\Exception
     */
    public function rebalance(KafkaConsumer $kafka, $err, array $partitions = null): void
    {
        switch ($err) {
            case RD_KAFKA_RESP_ERR__ASSIGN_PARTITIONS:
                $context = $this->buildLoggerContext($partitions);
                $this->logger->info('New partition assignment', $context);

                $kafka->assign($partitions);
                break;
            case RD_KAFKA_RESP_ERR__REVOKE_PARTITIONS:
                $context = $this->buildLoggerContext($kafka->getAssignment());
                $this->logger->info('Partitions revoked from consumer', $context);

                $kafka->assign(null);
                break;
            default:
                $this->logger->critical('Received invalid rebalance code', ['code' => $err]);
                throw new Exception(\sprintf('unexpected rebalance code: %d', $err));
        }
    }

    /**
     * @param null|TopicPartition[] $partitions
     *
     * @return mixed[]
     */
    private function buildLoggerContext(?array $partitions): array
    {
        $context = ['partitions' => []];
        if (!$partitions) {
            return $context;
        }

        foreach ($partitions as $part) {
            $context['partitions'][] = [
                'topic'  => $part->getTopic(),
                'part'   => $part->getPartition(),
                'offset' => $part->getOffset(),
            ];
        }

        return $context;
    }
}
