# About

This bundler provides easily configurable consumers and producers for Kafka, as well as
a transpor implementation to integrate with Symfony Messenger component.

# Configuration

You can find this configuration wth the command

```bash
$ php bin/console config:dump-reference cyber_misc
```

Relevant bundle configuration is as follows:

```yaml
cyber_misc:
    kafka:
        # List of broker configurations, used as config for topics
        configs:
            - # Prototype
                # All consumer with the same group.id will consume different partitions.
                group: ~ # Required
                # Comma delimited list of brokers host:port
                brokers: ~ # Required
                default_offset: ~ # One of "earliest"; "latest"; "error", Required
                # Enable automatic reporting of consumed messages. Must be off when used for transport.
                auto_offset_store: ~ # Required
                # Enable extra EndOfFile event for producers, else they simply block until timeout upon reaching end of messages.
                emit_eof: true
                # security protocol for brokers
                security_proto: ssl # One of "plaintext"; "ssl"; "sasl_plaintext"; "sasl_ssl"
                # Additional configs to pass to kafka; see: https://github.com/confluentinc/librdkafka/blob/master/CONFIGURATION.md
                extra: [ ]

        # List of topics to create consumers/producers for. Array keys is used as topic name.
        topics:
            - # Prototype
                # Name of the config from configs list to use for constructing consumer/producer pair for this topic.
                config: ~ # Required
                # Mark services as lazy
                lazy: false
                # Register topic with transport
                transport: false
                # Register topic as a streamer
                streamer: false

```

# Usage

## Provide Configs

First you must provide at least one set of configuration which defines communication parameters with kafka server.
The key of the `configs` array is used as the config name and will be used in other sections to refer to the
config parameters.

## Topics

Next create at least one topic configuration. Each key under `topics` will be used as the reference name to generate
auto-wiring injection tokens. Each topic configuration will generate a consumer/producer pair of services that can be
autowired with:

* `SingleTopicConsumer $refNameConsumer`
* `SingleTopicProducer $refNameProducer`

> If topic section name is `data` the injection token for consumer would be `SingleTopicConsumer $dataConsumer`

## Streaming Consumer

There is one additional service you can inject to make message consumption for workers easier. The `KafkaStreamer` is a
wrapper around the standard consumer which provides an "infinite" stream of messages via `\Generator`.

To activate a streamer you must set `streamer` option to true for each topic configuration where streamer is needed.
Then you can inject the streamer similar to producers:

* `KafkaStreamerInterface $refNameStreamer`

> If topic section name is `data` the injection token would be `KafkaStreamerInterface $dataStreamer`

### Streaming Events

Calling `streamEvents($idleTimeout)` returns a `\Generator` which does **NOT** complete until `stop()` is called.

If there are no messages for the specified `$idleTimeout` seconds the idle callbacks will be triggered. To register
such callbacks call `addIdleCallback($callback)`.
> Instance of the streamer will be passed as the sole argument to the callback.

To avoid memory leaks, it is recommended to have some mechanism to `stop()` the loop and quit the script, then
start it again. Some ideas:

* quit after x messages
* quit after x seconds
* quit whenever idle

### Failures

During streaming, messages are only marked as handled if the loop iteration completes without an early exit, thus the
only valid way to interrupt the loop is by calling `stop()` on the streamer instance. Things that cause early exist:

* using `break`
* throwing exception
* calling `exit` or `die`
* any other condition that causes script to terminate prematurely.

With this approach the messages will not get missed if a random error occurs during processing. However, at the same
time if a message is the cause of error, it will keep getting retried and failing until the code properly addresses
the error.

## Messenger Transport

Furthermore, each of the topics can be used for Symfony Messenger component. Configure in 2 simple steps:

1. set the `transport` flag for topic to true
2. direct the messenger to use kafka transport:

```yaml
framework:
    messenger:
        transports:
            kafka: 'kafka://topic_ref_name'
```

# Example Config

With the below config you can autowire the following 3 services to send and receive messages:

* `SingleTopicProducer $testTopicProducer'`
* `SingleTopicConsumer $testTopicConsumer`
* `KafkaStreamerInterface $testTopicStreamer`

```yaml
cyber_misc:
    kafka:
        configs:
            main:
                group: test
                brokers: kafka:9092
                default_offset: earliest
                auto_offset_store: false
                emit_eof: true
                security_proto: plaintext
        topics:
            test_topic:
                topic_name: test
                config: main
                transport: true
                streamer: true
```

Since the `test_topi` is marked with `transport: true` you can specify it in the Symfony Messenger's config as follows:

```yaml
framework:
    messenger:
        transports:
            kafka: 'kafka://test_topic'
```
