<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\MiscBundle\DependencyInjection;

use Cyber\MiscBundle\DependencyInjection\CyberMiscExtension;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * @internal
 *
 * @coversNothing
 */
class CyberMiscExtensionTest extends TestCase
{
    public function testEmptyLoad(): void
    {
        $extension = new CyberMiscExtension();
        $container = new ContainerBuilder();

        $extension->load([[]], $container);

        static::assertTrue(true); // if we got no exception we are good.
    }

    /**
     * @group skip7
     */
    public function testSomeConfigLoad(): void
    {
        $extension = new CyberMiscExtension();
        $container = new ContainerBuilder();

        $config = [
            'paginator_response_listener' => true,
            'kafka'                       => [
                'configs' => [
                    'main' => [
                        'group'             => '1',
                        'brokers'           => '2',
                        'default_offset'    => 'error',
                        'auto_offset_store' => true,
                        'emit_eof'          => true,
                        'security_proto'    => 'plaintext',
                    ],
                ],
                'topics'  => [
                    'sample' => ['config' => 'main', 'topic_name' => 'test'],
                ],
            ],
        ];

        $extension->load([$config], $container);
        // assert stuff

        static::assertTrue(true);
    }

    /**
     * @group skip7
     */
    public function testCompile(): void
    {
        $extension    = new CyberMiscExtension();
        $container    = new ContainerBuilder();
        $validConfigs = [];

        $extension->load([$validConfigs], $container);
        $container->compile();

        // assert container has some service
        static::assertTrue(true);
    }

    /**
     * @group skip7
     */
    public function testIllegalKafkaConfigs(): void
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage('When using kafka topic for transport, the "auto_offset_store", must be disabled. Topic: sample');
        $extension = new CyberMiscExtension();
        $container = new ContainerBuilder();

        $config = [
            'paginator_response_listener' => true,
            'kafka'                       => [
                'configs' => [
                    'main' => [
                        'group'             => '1',
                        'brokers'           => '2',
                        'default_offset'    => 'earliest',
                        'auto_offset_store' => true,
                        'emit_eof'          => true,
                        'security_proto'    => 'plaintext',
                    ],
                ],
                'topics'  => [
                    'sample' => ['config' => 'main', 'transport' => true, 'topic_name' => 'test'],
                ],
            ],
        ];

        $extension->load([$config], $container);
        // assert stuff

        static::assertTrue(true);
    }

    /**
     * @group skip7
     */
    public function testIllegalKafkaTopicConfigs(): void
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage('Broker config with name "mainx" is not found for topic: sample');
        $extension = new CyberMiscExtension();
        $container = new ContainerBuilder();

        $config = [
            'paginator_response_listener' => true,
            'kafka'                       => [
                'configs' => [
                    'main' => [
                        'group'             => '1',
                        'brokers'           => '2',
                        'default_offset'    => 'latest',
                        'auto_offset_store' => true,
                        'emit_eof'          => true,
                        'security_proto'    => 'plaintext',
                    ],
                ],
                'topics'  => [
                    'sample' => ['config' => 'mainx', 'topic_name' => 'test'],
                ],
            ],
        ];

        $extension->load([$config], $container);
        // assert stuff

        static::assertTrue(true);
    }

    /**
     * @group skip7
     */
    public function testIllegalStreamerConfigs(): void
    {
        $this->expectException(InvalidConfigurationException::class);
        $this->expectExceptionMessage('When using kafka topic for streamer, the "auto_offset_store", must be disabled. Topic: sample');
        $extension = new CyberMiscExtension();
        $container = new ContainerBuilder();

        $config = [
            'paginator_response_listener' => true,
            'kafka'                       => [
                'configs' => [
                    'main' => [
                        'group'             => '1',
                        'brokers'           => '2',
                        'default_offset'    => 'earliest',
                        'auto_offset_store' => true,
                        'emit_eof'          => true,
                        'security_proto'    => 'plaintext',
                    ],
                ],
                'topics'  => [
                    'sample' => ['config' => 'main', 'streamer' => true, 'topic_name' => 'test'],
                ],
            ],
        ];

        $extension->load([$config], $container);
        // assert stuff

        static::assertTrue(true);
    }
}
