<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\MiscBundle\Kafka;

use Psr\Log\LoggerInterface;
use RdKafka\Message;

if (!\defined('RD_KAFKA_RESP_ERR__AUTO_OFFSET_RESET')) {
    // polyfill
    \define('RD_KAFKA_RESP_ERR__AUTO_OFFSET_RESET', -140);
}

trait KafkaConsumerHelperTrait
{
    private LoggerInterface $logger;

    private function preProcessMessage(?Message $message): ?Message
    {
        static $loggedTimeout = false;

        if (!$message) {
            return null;
        }

        $context = ['data' => $message];

        switch ($message->err) {
            case RD_KAFKA_RESP_ERR_NO_ERROR:
                // good message
                $this->logger->debug('Received a new message.', $context);
                $loggedTimeout = false;

                return $message;
            case RD_KAFKA_RESP_ERR__PARTITION_EOF:
                $this->logger->debug('Reached end of partition.', $context);

                return null;
            case RD_KAFKA_RESP_ERR__AUTO_OFFSET_RESET:
                $this->logger->critical(
                    'Got offset reset error. This typically happens when "default_offset" is set to "error" and no initial offset was specified.',
                    $context
                );

                return null;
            case RD_KAFKA_RESP_ERR__TIMED_OUT:
                if (!$loggedTimeout) {
                    // prevent spam, as this can be triggered hundreds of times per second.
                    $this->logger->debug('Timeout occurred while waiting for next message.', $context);
                    $loggedTimeout = true;
                }

                return null;
            default:
                $this->logger->critical('Unexpected error has occured', $context);

                return null;
        }
    }
}
