<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\MiscBundle\EventListener;

use Nelmio\ApiDocBundle\OpenApiPhp\Util;
use OpenApi\Analysis;
use OpenApi\Annotations\Components;
use OpenApi\Annotations\OpenApi;
use OpenApi\Annotations\Operation;
use OpenApi\Annotations\Parameter;
use OpenApi\Annotations\Property;
use OpenApi\Annotations\Schema;
use OpenApi\Generator;

/**
 * @SuppressWarnings(PHPMD.StaticAccess) OpenApi uses statics
 */
class NelmioQueryParamProcessor
{
    public const X_QUERY_AGS_REF = 'query-args-$ref';

    public function __construct()
    {
    }

    public function __invoke(Analysis $analysis): void
    {
        /** @var Operation[] $operations */
        $operations = $analysis->getAnnotationsOfType(Operation::class);
        /** @var OpenApi $openApi */
        $openApi = $analysis->openapi;
        foreach ($operations as $operation) {
            $originalParams = [];
            $replacements   = [];
            foreach ($operation->parameters as $parameter) {
                if (Generator::UNDEFINED !== $parameter->x && \array_key_exists(self::X_QUERY_AGS_REF, $parameter->x)) {

                    $configs = $parameter->x[self::X_QUERY_AGS_REF];
                    // this will handle the case where ref=Model on the schema itself.
                    $replacements[] = $this->processSchema(
                        $openApi,
                        $operation,
                        $parameter->schema,
                        $parameter->name,
                        $configs
                    );

                    if (\is_array($parameter->schema->properties)) {
                        // this will handle the case where schema has properties with ref=model
                        foreach ($parameter->schema->properties as $property) {
                            $replacements[] = $this->processSchema(
                                $openApi,
                                $operation,
                                $property,
                                $property->property,
                                $configs
                            );
                        }
                    }

                    $analysis->annotations->detach($parameter->schema);
                    $analysis->annotations->detach($parameter);
                    continue;
                }
                $originalParams[] = $parameter;
            }
            $operation->parameters = \array_merge($originalParams, ...$replacements);
        }
    }

    /**
     * Expand the given operation by injecting parameters for all properties of the given schema.
     *
     * @return Parameter[]
     */
    protected function expandQueryArgs(Operation $operation, Schema $schema, string $nestedName, mixed $configs): array
    {
        /** @var Generator::UNDEFINED|Property[] $schemaProps */
        $schemaProps = $schema->properties;
        if (Generator::UNDEFINED === $schemaProps || !$schema->properties) {
            return [];
        }

        $parameters = [];
        $required   = Generator::isDefault($schema->required) ? [] : \array_flip($schema->required);

        foreach ($schema->properties as $property) {
            $isNullable  = Generator::isDefault($property->nullable) ? !isset($required[$property->property]) : $property->nullable;
            $type        = Generator::isDefault($property->format) ? $property->type : $property->format;
            $explode     = Generator::UNDEFINED;
            $paramSchema = new Schema(
                [
                    'type'     => $type,
                    'items'    => $property->items,
                    'nullable' => $isNullable,
                    'enum'     => $property->enum,
                ]
            );

            $paramSchema->_context = $operation->_context; // inherit context from operation, required to pretend to be a parameter

            $propName = $this->buildPropName($nestedName, $property->property, $type, $configs);

            if ('array' === $type) {
                $explode = 'explode' === $configs;
            }

            $parameter           = new Parameter([
                'name'        => $propName,
                'in'          => 'query',
                'required'    => !$isNullable,
                'schema'      => $paramSchema,
                'explode'     => $explode,
                'description' => $this->parseDescription($property),
                'example'     => $property->example,
            ]);
            $parameter->_context = $operation->_context; // inherit context from operation, required to pretend to be a parameter

            $parameters[] = $parameter;
        }

        return $parameters;
    }

    /**
     * @param non-empty-array<string>|string $type
     */
    private function buildPropName(string $nestedName, string $propName, $type, mixed $configs): string
    {
        // if we are nested wrap in php style array nested[prop];
        $propName = $nestedName ? \sprintf('%s[%s]', $nestedName, $propName) : $propName;

        // further more if we are of type array and config requests explode, append a `[]` to make it work properly
        if ('array' === $type && 'explode' === $configs) {
            $propName .= '[]';
        }

        return $propName;
    }

    /**
     * @return Parameter[]
     */
    private function processSchema(
        OpenApi $openApi,
        Operation $operation,
        Schema $schema,
        string $nestedName,
        mixed $configs,
    ): array {
        if (\is_string($schema->ref) && Generator::UNDEFINED !== $schema->ref) {
            $name   = \str_replace(Components::SCHEMA_REF, '', $schema->ref);
            $schema = Util::getSchema($openApi, $name);

            return $this->expandQueryArgs($operation, $schema, $nestedName, $configs);
        }

        return [];
    }

    private function parseDescription(Property $property): string
    {
        $parts = [];
        if (!Generator::isDefault($property->title)) {
            $parts[] = \trim($property->title, '.');
        }
        if (!Generator::isDefault($property->description)) {
            $parts[] = '<br/><small>' . $property->description . '</small>';
        }
        if ($parts) {
            return \implode('. ', $parts);
        }

        return Generator::UNDEFINED;
    }
}
