<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Tests\Cyber\MiscBundle\Event\Subscriber;

use Cyber\MiscBundle\Event\Subscriber\PaginationSubscriber;
use Doctrine\ORM\QueryBuilder;
use FOS\RestBundle\View\View;
use Knp\Component\Pager\Pagination\PaginationInterface;
use Knp\Component\Pager\PaginatorInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Event\ViewEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * @internal
 *
 * @coversNothing
 */
class PaginationSubscriberTest extends TestCase
{
    /**
     * @var PaginationSubscriber
     */
    private $subscriber;

    /**
     * @var HttpKernelInterface|MockObject
     */
    private $kernel;

    /**
     * @var MockObject|RequestStack
     */
    private $stack;

    /**
     * @var MockObject|PaginatorInterface
     */
    private $paginator;

    protected function setUp(): void
    {
        $defaults = [
            'default_page'      => 1,
            'default_page_size' => 25,
            'min_page_size'     => 10,
            'max_page_size'     => 1000,
        ];

        $this->kernel     = $this->getMockBuilder(HttpKernelInterface::class)->getMock();
        $this->stack      = $this->getMockBuilder(RequestStack::class)->getMock();
        $this->paginator  = $this->getMockBuilder(PaginatorInterface::class)->getMock();
        $this->subscriber = new PaginationSubscriber($this->stack, $this->paginator, $defaults);
    }

    public function testPaginateViewWrapperResponse(): void
    {
        $request = new Request();

        $this->stack->expects(static::any())
            ->method('getMainRequest')
            ->willReturn($request);

        $builder    = $this->getMockBuilder(QueryBuilder::class)->disableOriginalConstructor()->getMock();
        $pagination = $this->getMockBuilder(PaginationInterface::class)->disableOriginalConstructor()->getMock();

        $this->paginator->expects(static::once())
            ->method('paginate')
            ->with($builder)
            ->willReturn($pagination);

        $view = new View($builder);

        $event = new ViewEvent(
            $this->kernel,
            $request,
            HttpKernelInterface::MASTER_REQUEST,
            $view
        );

        $this->subscriber->paginateResponse($event);

        static::assertSame($view, $event->getControllerResult());
    }

    public function testPaginateQbResponse(): void
    {
        $request = new Request();

        $this->stack->expects(static::any())
            ->method('getMainRequest')
            ->willReturn($request);

        $builder    = $this->getMockBuilder(QueryBuilder::class)->disableOriginalConstructor()->getMock();
        $patination = $this->getMockBuilder(PaginationInterface::class)->getMock();

        $this->paginator->expects(static::once())
            ->method('paginate')
            ->with($builder)
            ->willReturn($patination);

        $event = new ViewEvent(
            $this->kernel,
            $request,
            HttpKernelInterface::MASTER_REQUEST,
            $builder
        );

        $this->subscriber->paginateResponse($event);

        static::assertSame($patination, $event->getControllerResult());
    }

    public function testPaginateUnsupportedResponse(): void
    {
        $request  = new Request();
        $response = ['unsupported', 'response'];

        $this->stack->expects(static::any())
            ->method('getMainRequest')
            ->willReturn($request);

        $this->paginator->expects(static::never())
            ->method('paginate');

        $event = new ViewEvent(
            $this->kernel,
            $request,
            HttpKernelInterface::MASTER_REQUEST,
            $response
        );

        $this->subscriber->paginateResponse($event);

        static::assertSame($response, $event->getControllerResult());
    }
}
