<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Tests\Cyber\MiscBundle;

use ArrayIterator;
use Cyber\MiscBundle\SpreadsheetDataProcessor;
use PHPUnit\Framework\TestCase;
use stdClass;

/**
 * @covers \Cyber\MiscBundle\SpreadsheetDataProcessor
 *
 * @internal
 */
class SpreadsheetDataProcessorTest extends TestCase
{
    /**
     * @var SpreadsheetDataProcessor<array{id: string, data: string}>
     */
    private $instance;

    /**
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    protected function setUp(): void
    {
        $this->instance = new class () extends SpreadsheetDataProcessor {
            /**
             * @inheritDoc
             */
            protected function getColumnMapping(array $options): array
            {
                return [
                    'col 1'     => 'id',
                    'other col' => 'data',
                ];
            }

            /**
             * @inheritDoc
             */
            protected function mapRow(array &$data, int $line, array $options)
            {
                // convert to stdClass
                return (object) $data;
            }
        };
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException never
     */
    public function testProcessSuccess(): void
    {
        $data = [
            ['col 1', 'other col'],
            ['1', '2'],
            ['3', '4'],
            ['', '5'],
        ];

        /** @var stdClass[] $results */
        $results = \iterator_to_array($this->instance->process(new ArrayIterator($data)));

        static::assertCount(3, $results);
        static::assertContainsOnlyInstancesOf(stdClass::class, $results);
        static::assertEquals('1', $results[0]->id);
        static::assertEquals('2', $results[0]->data);
        static::assertEquals('3', $results[1]->id);
        static::assertEquals('4', $results[1]->data);
        static::assertEquals('', $results[2]->id);
        static::assertEquals('5', $results[2]->data);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException always
     */
    public function testProcessMissingOrInvalidColumns(): void
    {
        $this->expectException(\Cyber\MiscBundle\Exception\SheetProcessorException::class);
        $this->expectExceptionMessage('Report format does not match to expected');

        $data = [
            ['col 1', 'wrong column'],
            ['1', '2'],
            ['3', '4'],
        ];

        \iterator_to_array($this->instance->process(new ArrayIterator($data)));
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException never
     */
    public function testProcessExtraColumns(): void
    {
        $data = [
            ['col 1', 'other col', 'extra column'],
            ['1', '2', 'a'],
            ['3', '4', 'b'],
        ];

        /** @var stdClass[] $results */
        $results = \iterator_to_array($this->instance->process(new ArrayIterator($data)));

        static::assertCount(2, $results);
        static::assertContainsOnlyInstancesOf(stdClass::class, $results);
        static::assertEquals('a', $results[0]->unknown_column_2);
        static::assertEquals('b', $results[1]->unknown_column_2);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException never
     */
    public function testProcessIgnoreExtraValuesFlag(): void
    {
        $data = [
            ['col 1', 'other col'],
            ['1', '2', 'a'],
            ['3', '4', 'b'],
        ];

        $results = \iterator_to_array($this->instance->process(
            new ArrayIterator($data),
            [SpreadsheetDataProcessor::OPT_IGNORE_EXTRA_COLUMNS => true]
        ));

        static::assertCount(2, $results);
        static::assertContainsOnlyInstancesOf(stdClass::class, $results);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException never
     */
    public function testProcessIgnoreMissingValuesFlag(): void
    {
        $data = [
            ['col 1', 'other col'],
            ['1'],
            ['3'],
        ];

        /** @var stdClass[] $results */
        $results = \iterator_to_array($this->instance->process(
            new ArrayIterator($data),
            [SpreadsheetDataProcessor::OPT_IGNORE_MISSING_COLUMNS => true]
        ));

        static::assertCount(2, $results);
        static::assertContainsOnlyInstancesOf(stdClass::class, $results);
        static::assertEquals('1', $results[0]->id);
        static::assertEquals('', $results[0]->data);
        static::assertEquals('3', $results[1]->id);
        static::assertEquals('', $results[1]->data);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException always
     */
    public function testProcessFailOnExtraValues(): void
    {
        $this->expectException(\Cyber\MiscBundle\Exception\SheetProcessorException::class);
        $this->expectExceptionMessage('Expected 2, but got 3');

        $data = [
            ['col 1', 'other col'],
            ['1', '2', 'a'],
            ['3', '4', 'b'],
        ];

        \iterator_to_array($this->instance->process(new ArrayIterator($data)));
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException always
     */
    public function testProcessFailOnMissingValues(): void
    {
        $this->expectException(\Cyber\MiscBundle\Exception\SheetProcessorException::class);
        $this->expectExceptionMessage('Expected 2, but got 1');

        $data = [
            ['col 1', 'other col'],
            ['1'],
            ['3'],
        ];

        \iterator_to_array($this->instance->process(new ArrayIterator($data)));
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException always
     */
    public function testProcessEmptyIterator(): void
    {
        $this->expectException(\Cyber\MiscBundle\Exception\SheetProcessorException::class);
        $this->expectExceptionMessage('No data in the sheet');

        $data = [];

        \iterator_to_array($this->instance->process(new ArrayIterator($data)));
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function testDefaultMap(): void
    {
        $this->instance = new class () extends SpreadsheetDataProcessor {
            /**
             * @inheritDoc
             */
            protected function getColumnMapping(array $options): array
            {
                return [
                    'col 1'     => 'id',
                    'other col' => 'data',
                ];
            }
        };

        $data = [
            ['col 1', 'other col'],
            ['1', '2'],
            ['3', '4'],
        ];

        /** @var array<mixed> $results */
        $results = [];
        foreach ($this->instance->process(new ArrayIterator($data)) as $row) {
            $results[] = $row;
        }
        static::assertCount(2, $results);
        static::assertEquals('1', $results[0]['id']);
        static::assertEquals('2', $results[0]['data']);
        static::assertEquals('3', $results[1]['id']);
        static::assertEquals(4, $results[1]['data']);
    }
}
