<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\Spec;

use Cyber\OrmExtras\Spec;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\QueryBuilder;
use InvalidArgumentException;

/**
 * @method string getEntityName()
 * @method QueryBuilder createQueryBuilder(string $alias)
 *
 * @property string $defaultAlias
 */
trait EntitySpecRepoTrait
{
    /**
     * @param Spec $specification
     *
     * @throws NonUniqueResultException
     *
     * @return mixed
     */
    public function matchSpec(Spec $specification)
    {
        $query  = $this->matchSpecQuery($specification);
        $result = (array) $query->execute();

        if ($query->hasHint(Spec::HINT_SINGLE_RESULT)) {
            $first = \array_shift($result);
            if (!empty($result)) {
                throw new NonUniqueResultException('The Spec::HINT_SINGLE_RESULT hint is set, but multiple rows were returned');
            }

            return $first;
        }

        return $result;
    }

    public function matchSpecQuery(Spec $specification): AbstractQuery
    {
        /** @var class-string $entityClass */
        $entityClass = $this->getEntityName();

        if (!$specification->supports($entityClass)) {
            throw new InvalidArgumentException(\sprintf(
                '%s does not support %s in %s.',
                \get_class($specification),
                $entityClass,
                static::class
            ));
        }

        $qb   = $this->createQueryBuilder($this->getDefaultAlias());
        $expr = $specification->match($qb, $this->getDefaultAlias());
        if (null !== $expr) {
            // maybe spec only does modify query, so it is fine for it not return an expression
            // avoids pollution with 1=1 conditions
            $qb->where($expr);
        }
        $query = $qb->getQuery();

        $specification->modifyQuery($query);

        return $query;
    }

    protected function getDefaultAlias(): string
    {
        return \property_exists(__CLASS__, 'defaultAlias') ? $this->defaultAlias : 'spec';
    }
}
