<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\Spec;

use Doctrine\ORM\QueryBuilder;
use LogicException;

abstract class JoiningSpec extends ChildSpec
{
    /**
     * @var string the main alias of the join. It will be passed to parent spec's match() call.
     */
    protected $joinAlias;

    /**
     * @var class-string the class of the join relationship. It will be passed to parent spec's supports() call.
     */
    protected $joinClass;

    /**
     * Replacement for the match() call for join relationship.
     *
     * This class does not allow you to override the default match() method, instead this one
     * is called from within match() with same parameters that you would normally get in match().
     *
     * @param QueryBuilder $qb
     * @param string       $dqlAlias
     *
     * @see Spec::match()
     */
    abstract public function joinMatch(QueryBuilder $qb, string $dqlAlias): void;

    /**
     * Replacement for the supports() call for join relationship.
     *
     * This class does not allow you to override the default supports() method, instead this one
     * is called from within supports() with same parameters that you would normally get in supports().
     *
     * @param class-string $className
     *
     * @return bool
     *
     * @see Spec::supports()
     */
    abstract public function joinSupports(string $className): bool;

    /**
     * {@inheritdoc}
     */
    final public function match(QueryBuilder $qb, string $dqlAlias)
    {
        $this->joinMatch($qb, $dqlAlias);

        if (!$this->joinAlias) {
            throw new LogicException(\sprintf('The "joinAlias" property of "%s" must be defined.', static::class));
        }

        return parent::match($qb, $this->joinAlias);
    }

    /**
     * {@inheritdoc}
     */
    final public function supports(string $className): bool
    {
        if (!$this->joinClass) {
            throw new LogicException(\sprintf('The "joinClass" property of "%s" must be defined.', static::class));
        }

        return $this->joinSupports($className) && parent::supports($this->joinClass);
    }
}
