<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\OrmExtras\Functions;

use Cyber\OrmExtras\Functions\MatchAgainst;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\ORM\Configuration;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Parser;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

class MatchAgainstTest extends TestCase
{
    /**
     * @dataProvider dataProvider
     *
     * @param mixed $dql
     * @param mixed $sql
     */
    public function test($dql, $sql): void
    {
        $config = new Configuration();
        $config->addCustomStringFunction('match', MatchAgainst::class);

        $platform = $this->getMockBuilder(AbstractPlatform::class)
            ->disableOriginalConstructor()
            ->getMock();

        if (\method_exists(AbstractPlatform::class, 'getSQLResultCasing')) {
            $platform->expects($this->any())
                ->method('getSQLResultCasing')
                ->willReturn('sclr_0');
        }

        $platform->expects($this->any())
            ->method('quoteIdentifier')
            ->willReturnArgument(0);

        $platform->expects($this->any())
            ->method('appendLockHint')
            ->willReturnArgument(0);

        $con = $this->getMockBuilder(Connection::class)
            ->disableOriginalConstructor()
            ->getMock();

        $con->expects($this->any())
            ->method('getDatabasePlatform')
            ->willReturn($platform);

        /** @var ClassMetadata<object>|MockObject $metadata */
        $metadata = $this->getMockBuilder(ClassMetadata::class)
            ->disableOriginalConstructor()
            ->getMock();

        $metadata->table['name']       = 'person';
        $metadata->fieldMappings['id'] = [
            'fieldName'  => 'id',
            'type'       => 'string',
            'columnName' => 'prs_id',
            'options'    => [],
        ];

        $metadata->expects($this->any())
            ->method('getTableName')
            ->willReturn('person');

        /** @var EntityManagerInterface|MockObject $em */
        $em = $this->getMockBuilder(EntityManagerInterface::class)
            ->getMock();
        $em->expects($this->any())
            ->method('getConfiguration')
            ->willReturn($config);

        $em->expects($this->any())
            ->method('getClassMetadata')
            ->willReturn($metadata);

        $em->expects($this->any())
            ->method('getConnection')
            ->willReturn($con);

        $query = new Query($em);
        $query->setDQL($dql);

        $parser = new Parser($query);

        $statements = $parser->parse()->getSqlExecutor()->getSqlStatements();

        $this->assertEquals($sql, $statements);
    }

    /**
     * @return \Generator<array<string>>
     */
    public function dataProvider(): \Generator
    {
        yield 'test#1 no additional parameters' => [
            'dql' => 'SELECT MATCH(person.id) AGAINST (:againstParam) from \Tests\Cyber\OrmExtras\Fixtures\PersonEntity AS person',
            'sql' => 'SELECT MATCH (p0_.prs_id) AGAINST (?) AS sclr_0 FROM person p0_',
        ];

        yield 'test#2 Boolean Full-Text Searches' => [
            'dql' => 'SELECT MATCH(person.id) AGAINST (:againstParam BOOLEAN) from \Tests\Cyber\OrmExtras\Fixtures\PersonEntity AS person',
            'sql' => 'SELECT MATCH (p0_.prs_id) AGAINST (? IN BOOLEAN MODE) AS sclr_0 FROM person p0_',
        ];

        yield 'test#3 with Query Expansion Searches' => [
            'dql' => 'SELECT MATCH(person.id) AGAINST (:againstParam EXPAND) from \Tests\Cyber\OrmExtras\Fixtures\PersonEntity AS person',
            'sql' => 'SELECT MATCH (p0_.prs_id) AGAINST (? WITH QUERY EXPANSION) AS sclr_0 FROM person p0_',
        ];

        yield 'test#3 Boolean Full-Text with Query Expansion Searches' => [
            'dql' => 'SELECT MATCH(person.id) AGAINST (:againstParam BOOLEAN EXPAND) from \Tests\Cyber\OrmExtras\Fixtures\PersonEntity AS person',
            'sql' => 'SELECT MATCH (p0_.prs_id) AGAINST (? IN BOOLEAN MODE WITH QUERY EXPANSION) AS sclr_0 FROM person p0_',
        ];
    }
}
