<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\Event;

use Cyber\OrmExtras\Supplier\CurrentUserSupplier;
use Cyber\OrmExtras\Utility\CreatorAttributes;
use Cyber\OrmExtras\Utility\EditorAttributes;
use DateTime;
use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Event\OnFlushEventArgs;
use Doctrine\ORM\UnitOfWork;

class CreatorEditorLifecycleSubscriber implements EventSubscriber
{
    /** @var array<bool> */
    private $processedRelations = [];

    /** @var EntityManagerInterface */
    private $em;

    /** @var UnitOfWork */
    private $uow;

    /** @var CurrentUserSupplier */
    private $userSupplier;

    public function __construct(CurrentUserSupplier $supplier)
    {
        $this->userSupplier = $supplier;
    }

    /**
     * @return string[]
     */
    public function getSubscribedEvents(): array
    {
        return [
            'onFlush',
        ];
    }

    /**
     * @param OnFlushEventArgs $eventArgs
     *
     * @throws \Doctrine\ORM\ORMInvalidArgumentException
     * @throws \Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException
     * @throws \Symfony\Component\DependencyInjection\Exception\ServiceCircularReferenceException
     */
    public function onFlush(OnFlushEventArgs $eventArgs): void
    {
        $this->processedRelations = []; //reset the array in case we get multiple flushes
        $this->em                 = $eventArgs->getEntityManager();
        $this->uow                = $this->em->getUnitOfWork();

        foreach ($this->uow->getScheduledEntityInsertions() as $entity) {
            $this->processCreatorAttributes($entity);
        }

        foreach ($this->uow->getScheduledEntityUpdates() as $entity) {
            $this->processEditorAttributes($entity);
        }

        foreach ($this->uow->getScheduledEntityDeletions() as $entity) {
            $this->processRelatedEntities($entity);
        }
    }

    /**
     * @param mixed $entity
     *
     * @throws \Doctrine\ORM\ORMInvalidArgumentException
     * @throws \Symfony\Component\DependencyInjection\Exception\ServiceCircularReferenceException
     * @throws \Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException
     */
    private function processCreatorAttributes($entity): void
    {
        if ($entity instanceof CreatorAttributes) {
            if (null === $entity->getUserCreated()) {
                $entity->setUserCreated($this->userSupplier->getCurrentUser());
            }
            if (null === $entity->getDateCreated()) {
                $entity->setDateCreated(new DateTime());
            }
            $this->computeChangeSet($entity);
        }

        $this->processEditorAttributes($entity);
    }

    /**
     * @param mixed $entity
     *
     * @throws \Doctrine\ORM\ORMInvalidArgumentException
     */
    private function computeChangeSet($entity): void
    {
        $meta = $this->em->getClassMetadata(\get_class($entity));
        $this->uow->recomputeSingleEntityChangeSet($meta, $entity);
    }

    /**
     * @param mixed $entity
     *
     * @throws \Doctrine\ORM\ORMInvalidArgumentException
     * @throws \Symfony\Component\DependencyInjection\Exception\ServiceCircularReferenceException
     * @throws \Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException
     */
    private function processEditorAttributes($entity): void
    {
        if ($entity instanceof EditorAttributes) {
            $entity->setUserEdited($this->userSupplier->getCurrentUser());
            $entity->setDateEdited(new DateTime());
            $this->computeChangeSet($entity);
        }

        $this->processRelatedEntities($entity);
    }

    /**
     * @param mixed $entity
     *
     * @throws \Doctrine\ORM\ORMInvalidArgumentException
     * @throws \Symfony\Component\DependencyInjection\Exception\ServiceCircularReferenceException
     * @throws \Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException
     */
    private function processRelatedEntities($entity): void
    {
        $relatedEntities = [];
        if ($entity instanceof CreatorAttributes) {
            $relatedEntities = \array_merge($entity->getCreatorCascade(), $relatedEntities);
        }
        if ($entity instanceof EditorAttributes) {
            $relatedEntities = \array_merge($entity->getEditorCascade(), $relatedEntities);
        }

        foreach ($relatedEntities as $relatedEntity) {
            if ($relatedEntity instanceof EditorAttributes) {
                $this->updateRelatedEditorAttrs($relatedEntity);
            }
        }
    }

    private function updateRelatedEditorAttrs(EditorAttributes $relatedEntity): void
    {
        $hash = \spl_object_hash($relatedEntity);
        if (isset($this->processedRelations[$hash])) {
            //entity was already updated probably due to it being linked to another entity within this flush.
            return;
        }

        $relatedEntity->setUserEdited($this->userSupplier->getCurrentUser());
        $relatedEntity->setDateEdited(new DateTime());
        $this->computeChangeSet($relatedEntity);
        $this->processRelatedEntities($relatedEntity);

        $this->processedRelations[$hash] = true;
    }
}
