<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\Doctrine;

use Cyber\OrmExtras\Annotations\TablePrefix;
use Doctrine\Common\Annotations\Reader;
use Doctrine\ORM\Mapping\UnderscoreNamingStrategy;
use ReflectionClass;
use UnexpectedValueException;

class PrefixNamingStrategy extends UnderscoreNamingStrategy
{
    /** @var null|Reader */
    private $reader;

    /** @var string[] */
    private $classNamePrefixes;

    public function __construct(?Reader $reader)
    {
        // todo this could be BC break
        parent::__construct(CASE_LOWER, true);
        $this->reader = $reader;
    }

    /**
     * @param string            $propertyName
     * @param null|class-string $className
     *
     * @throws \ReflectionException
     *
     * @return string
     */
    public function propertyToColumnName($propertyName, $className = null): string
    {
        return $this->convertFieldName($propertyName, $className);
    }

    /**
     * @param string            $propertyName
     * @param null|class-string $className
     *
     * @throws \ReflectionException
     *
     * @return string
     */
    public function joinColumnName($propertyName, $className = null): string
    {
        return $this->convertFieldName($propertyName, $className, $this->referenceColumnName());
    }

    /**
     * Find prefix for className.
     *
     * @param null|class-string $className
     *
     * @throws \ReflectionException
     *
     * @return string
     */
    public function findPrefix($className): string
    {
        if (null === $className) {
            return '';
        }

        if (isset($this->classNamePrefixes[$className])) {
            return $this->classNamePrefixes[$className];
        }

        /** @var null|TablePrefix $annotation */
        $annotation = $this->findAnnotation($className);

        return $this->classNamePrefixes[$className] = $annotation ? $annotation->getPrefix() : '';
    }

    /**
     * Converts FieldName according with ITSM table naming policy.
     *
     * @param string            $propertyName
     * @param null|class-string $className
     * @param string            $referenceColumn
     *
     * @throws \ReflectionException
     *
     * @return string
     */
    private function convertFieldName(string $propertyName, $className = null, string $referenceColumn = null): string
    {
        $prefix = $this->findPrefix($className);

        if ($referenceColumn) {
            $propertyName = $propertyName . '_' . $referenceColumn;
        }

        $propertyName = \preg_replace('~(?<=\\w)([A-Z])~', '_$1', $propertyName);

        \assert(
            null !== $propertyName,
            new UnexpectedValueException('Could not extrapolate field name for property: ' . $propertyName)
        );

        $propertyName = \mb_strtolower($propertyName);

        return ($prefix ? $prefix . '_' : '') . $propertyName;
    }

    /**
     * @param class-string $className
     */
    private function findAnnotation(string $className): ?TablePrefix
    {
        $ref = new ReflectionClass($className);

        if (null === $this->reader) {
            // we are using attribute driver
            $attribute = $ref->getAttributes(TablePrefix::class);

            return isset($attribute[0]) ? $attribute[0]->newInstance() : null;
        }

        return $this->reader->getClassAnnotation($ref, TablePrefix::class);
    }
}
