<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\DependencyInjection;

use Cyber\OrmExtras\Supplier\TokenStorageUserSupplier;
use Ramsey\Uuid\Guid\Guid;
use Ramsey\Uuid\Lazy\LazyUuidFromString;
use Ramsey\Uuid\Rfc4122\UuidV4;
use Ramsey\Uuid\Uuid;
use Symfony\Component\Config\Definition\Builder\ArrayNodeDefinition;
use Symfony\Component\Config\Definition\Builder\TreeBuilder;
use Symfony\Component\Config\Definition\ConfigurationInterface;

/**
 * This is the class that validates and merges configuration from your app/config files.
 *
 * To learn more see
 * {@link http://symfony.com/doc/current/cookbook/bundles/extension.html#cookbook-bundles-extension-config-class}
 */
class Configuration implements ConfigurationInterface
{
    /**
     * @inheritdoc
     *
     * @noinspection NullPointerExceptionInspection
     */
    public function getConfigTreeBuilder(): TreeBuilder
    {
        $treeBuilder = new TreeBuilder('cyber_orm_extras');

        /** @var ArrayNodeDefinition $rootNode */
        $rootNode = $treeBuilder->getRootNode();

        //@formatter:off
        $rootNode
            ->children()
                ->scalarNode('user_supplier')->cannotBeEmpty()->defaultValue(TokenStorageUserSupplier::class)->end()
                ->scalarNode('user_class')->isRequired()->cannotBeEmpty()->end()
                ->scalarNode('user_id_field')->cannotBeEmpty()->defaultValue('id')->end()
                ->scalarNode('default_user_id')->isRequired()->cannotBeEmpty()->end()
                ->arrayNode('allowed_classes')
                    ->info('List of classes your application uses inside of entity ids.')
                    ->defaultValue([UuidV4::class, Uuid::class, LazyUuidFromString::class, Guid::class])
                    ->scalarPrototype()
                    ->end()
                ->end()
                ->enumNode('metadata_driver')->values(['annotation', 'attribute'])->defaultValue('annotation')
                    ->info('What will you use to provide TablePrefix configs to entities, annotations or attributes')
                    ->validate()
                        ->always(function ($val) {
                            if ('annotation' === $val && PHP_VERSION_ID >= 80000) {
                                trigger_deprecation(
                                    'cyber/audit-bundle',
                                    '6.1',
                                    'Using annotation metadata_driver is deprecated since version 6.1, switch to Attributes'
                                );
                            }

                            return $val;
                        })
                    ->end()
                ->end()
            ->end();
        //@formatter:on
        return $treeBuilder;
    }
}
