<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\Functions;

use Doctrine\ORM\Query\AST\Functions\FunctionNode;
use Doctrine\ORM\Query\Lexer;
use Doctrine\ORM\Query\Parser;
use Doctrine\ORM\Query\QueryException;
use Doctrine\ORM\Query\SqlWalker;
use Doctrine\ORM\Query\TokenType;

/**
 * Provides a way to access an entity's discriminator field in DQL
 * queries.
 *
 * Assuming the same "Person" entity from Doctrine's documentation on
 * Inheritence Mapping, which has a discriminator field named "discr":
 *
 * Using the TYPE() function, DQL will interpret this:
 *
 * <pre>'SELECT TYPE(p) FROM Person p'</pre>
 *
 * as if you had written this:
 *
 * <pre>'SELECT p.discr FROM Person p'</pre>
 *
 * This conversion happens at the SQL level, so the ORM is no longer
 * part of the picture at that point.
 *
 * Normally, if you try to access the discriminator field in a DQL
 * Query, Doctrine will complain that the field does not exist on the
 * entity. This makes sense from an ORM point-of-view, but having
 * access to the discriminator field allows us to, for example:
 *
 * - get the type when we only have an ID
 * - query within a subset of all the available types
 */
class TypeFunction extends FunctionNode
{
    /**
     * @var string
     */
    public $dqlAlias;

    /**
     * @param SqlWalker $sqlWalker
     *
     * @throws QueryException
     *
     * @return string
     *
     * @SuppressWarnings(PHPMD.StaticAccess)
     */
    public function getSql(SqlWalker $sqlWalker): string
    {
        $class      = $sqlWalker->getMetadataForDqlAlias($this->dqlAlias);
        $tableAlias = $sqlWalker->getSQLTableAlias($class->getTableName(), $this->dqlAlias);
        if (!isset($class->discriminatorColumn['name'])) {
            throw QueryException::semanticalError(
                'TYPE() only supports entities with a discriminator column.'
            );
        }

        return $tableAlias . '.' . $class->discriminatorColumn['name'];
    }

    /**
     * @param Parser $parser
     *
     * @throws QueryException
     */
    public function parse(Parser $parser): void
    {
        // ORM v3 moved constants to TokenType
        $constRef = \enum_exists(TokenType::class) ? TokenType::class : Lexer::class;

        $parser->match($constRef::T_IDENTIFIER);
        $parser->match($constRef::T_OPEN_PARENTHESIS);
        $this->dqlAlias = $parser->IdentificationVariable();
        $parser->match($constRef::T_CLOSE_PARENTHESIS);
    }
}
