<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\Utility;

use Doctrine\ORM\Mapping as ORM;
use JMS\Serializer\Annotation as Serializer;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use RuntimeException;

trait UuidEntityTrait
{
    /**
     * ID of the record.
     *
     * @var UuidInterface
     */
    #[ORM\Id]
    #[ORM\Column(type: 'uuid')]
    #[Serializer\Accessor(getter: 'toStringId', setter: 'idFromString')]
    #[Serializer\Type('string')]
    protected UuidInterface $id;

    /**
     * Generates value for $id field.
     *
     * You must call this in the constructor of your entity.
     *
     * @param null|callable $callable a custom callable for uuid generation, intended for tests
     */
    private function generateId($callable = null): void
    {
        try {
            /** @var UuidInterface $uuid */
            $uuid     = \call_user_func($callable ?: [Uuid::class, 'uuid4']);
            $this->id = $uuid;
        } catch (\Exception $ex) {
            throw new RuntimeException('UUID generation failure, probably missing dependency.', 0, $ex);
        }
    }

    /**
     * @return UuidInterface the id of the entity
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * Use for serializing the data, as getId returns an object.
     *
     * @return string
     */
    public function toStringId(): string
    {
        return $this->getId()->toString();
    }

    /**
     * Use when de-serializing the data.
     *
     * @SuppressWarnings(PHPMD.StaticAccess)
     */
    public function idFromString(string $uuid): self
    {
        $this->id = Uuid::fromString($uuid);

        return $this;
    }
}
