<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\OrmExtras\Spec;

use Cyber\OrmExtras\Spec;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use PHPUnit\Framework\TestCase;
use Tests\Cyber\OrmExtras\Mock\BasicEntitySpecRepo;
use Tests\Cyber\OrmExtras\Mock\SpecRepoWithDefault;
use Tests\Cyber\OrmExtras\Mock\SpecRepoWithoutDefault;

/**
 * @covers \Cyber\OrmExtras\Spec\EntitySpecRepoTrait
 *
 * Query was final during some time, so skip the tests on lowest
 *
 * @group  skipLowest
 */
class EntitySpecRepoTraitTest extends TestCase
{
    public function testDefaultAlias(): void
    {
        $anonymousClass = new SpecRepoWithDefault();

        $this->assertEquals('test', $anonymousClass->getAlias());

        $anonymousClass = new SpecRepoWithoutDefault();

        $this->assertEquals('spec', $anonymousClass->getAlias());
    }

    /**
     * @throws NonUniqueResultException
     */
    public function testUnsupportedSpec(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('does not support some-name');

        $spec = $this->getMockBuilder(Spec::class)->getMock();
        $spec->method('supports')->willReturn(false);

        $qb = $this->getMockBuilder(QueryBuilder::class)->disableOriginalConstructor()->getMock();

        $repo = new BasicEntitySpecRepo($qb, 'some-name');
        $repo->matchSpec($spec);
    }

    public function testMatchSpec(): void
    {
        $query = $this->getMockBuilder(Query::class)->disableOriginalConstructor()->getMock();
        $query->expects($this->exactly(3))
            ->method('execute')
            ->willReturnOnConsecutiveCalls(
                [],
                ['single-result'],
                ['double', 'result'],
            );
        $query->expects($this->exactly(0))
            ->method('hasHint');

        $qb = $this->getMockBuilder(QueryBuilder::class)->disableOriginalConstructor()->getMock();
        $qb->method('where')->willReturnSelf();
        $qb->method('getQuery')->willReturn($query);

        $spec = $this->getMockBuilder(Spec::class)->getMock();
        $spec->method('supports')->willReturn(true);

        $repo = new BasicEntitySpecRepo($qb, 'some-name');

        $result = $repo->matchSpec($spec);
        $this->assertEquals([], $result);

        $result = $repo->matchSpec($spec);
        $this->assertEquals(['single-result'], $result);

        $result = $repo->matchSpec($spec);
        $this->assertEquals(['double', 'result'], $result);
    }

    public function testMatchSingleSpec(): void
    {
        $query = $this->getMockBuilder(Query::class)->disableOriginalConstructor()->getMock();
        $query->expects($this->exactly(2))
            ->method('execute')
            ->willReturnOnConsecutiveCalls(
                ['single-result'],
                ['double', 'result'],
            );

        $qb = $this->getMockBuilder(QueryBuilder::class)->disableOriginalConstructor()->getMock();
        $qb->method('where')->willReturnSelf();
        $qb->method('getQuery')->willReturn($query);

        $spec = $this->getMockBuilder(Spec::class)->getMock();
        $spec->method('supports')->willReturn(true);

        $repo = new BasicEntitySpecRepo($qb, 'some-name');

        $result = $repo->matchSingleSpec($spec);
        $this->assertEquals('single-result', $result);

        $this->expectException(NonUniqueResultException::class);
        $this->expectExceptionMessage('Single result requested, but multiple rows were returned');
        $repo->matchSingleSpec($spec);
    }
}
