Soft Deletable interface and filter allows to soft delete entities that will result in them being hidden from all selects;
and provide ability to show soft deleted entities by disabling the filter.

## Configuration

For this feature to work a proper [user supplier configuration](user-supplier.md) should be provided.

In addition you will most probably want to configure the doctrine filter to hide all soft deleted entities:

```yaml
doctrine:
    orm:
        filters:
            soft_deleted:
                class: Cyber\OrmExtras\Filters\SoftDeletedFilter
                enabled: true
```

## Usage

To use the Soft Deletable mechanism, implement the `SoftDeletable` interface. You may also want to use the provided 
`SoftDeletableTrait` to avoid having to repeatedly define same getters/setters.

Here is an example:

```php
<?php
use Cyber\OrmExtras\Utility\SoftDeletable;
use Cyber\OrmExtras\Utility\SoftDeletableTrait;

class PersonEntity implements SoftDeletable
{
    use SoftDeletableTrait;

    
    /**
     * @ORM\ManyToOne(targetEntity="YourBundle\Entity\User")    
     * @ORM\Column(name="user_deleted", referencedColumnName="Usr_Id", nullable=true)
     */
    protected $userDeleted;

    /**
     * @ORM\Column(name="date_deleted", type="datetime", nullable=true)
     */
    protected $dateDeleted;

}

```

The field names and corresponding getters/setters are provided by the traits. The traits also satisfy requirements of
the corresponding interfaces. So the only thing you really have to do is provide the `ORM` mapping for the columns.

### Soft Deleting

Once the entity implements `SoftDeletable` passing it to `$em->remove()` function will by default soft delete it.
To hard delete such entity you need to call `setForceDelete(true)` before passing it to `$em->remove()`.

```php
<?php
    $entity = $em->find(PersonEntity::class, 1);
    $em->remove($entity);
    $em->flush(); // entity is soft deleted
    
    $entity = $em->find(PersonEntity::class, 2);
    $entity->setForceDelete(true);
    $em->remove($entity);
    $em->flush(); // entity is hard deleted
```

Attempting to remove an entity, which has been already soft deleted from before, will also result in hard deletion.

```php
<?php
    $entity = $em->find(PersonEntity::class, 1); // fetch entity with id 1
    // assuming this entity was previously soft deleted
    $em->remove($entity);
    $em->flush(); //entity is now hard deleted
```

> This mechanism is actually based on simply evaluating if **dateDeleted** field has a non `null` value; so technically,
> as an alternative way you, could set **dateDeleted** to some value to trigger hard delete instead of using 
> `setForceDelete()` function. *#options*

### Fetching Soft Deleted Entities

If you ever want to select the soft deleted entities you have 2 options:

1. disable the filter completely
2. disable the filter for specific entity

#### Disable filter completely

To achieve this, you need to tell entity manager that you wish to disable a particular filter.

```php
<?php
    $em->getFilters()->disable('soft_deleted'); //the name should match what you have specified in config
    //do your selects
    $em->getFilters()->enable('soft_deleted');
```
> keep in mind that once filter is disabled it will affect all subsequent queries. So you might want to re-enable the
> filter after you are done fetching your soft deleted entities (just in case if there is other code after yours which 
> performs more selects and expects the filter to be in its enabled state)

#### Disable filter for entity
To achieve this, you need to get the filter from the entity manager and tell it which filter to disable. In the example
below we disable the filter for `PersonEntity` to fetch the soft deleted entity

```php
<?php
    /** @var SoftDeletedFilter $filter */
    $filter = $em->getFilters()->getFilter('soft_deleted'); //the name should match what you have specified in config
    $filter->disableForEntity(PersonEntity::class); 
    
    $entity = $em->find(PersonEntity::class, 1);
    
    $filter->enableForEntity(PersonEntity::class);
```

> keep in mind that once filter is disabled it will affect all subsequent queries. So you might want to re-enable the
> filter after you are done fetching your soft deleted entities (just in case if there is other code after yours which 
> performs more selects and expects the filter to be in its enabled state)
