<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\OrmExtras\Spec;

use Cyber\OrmExtras\Spec;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query\Expr;
use Doctrine\ORM\QueryBuilder;

class AndX implements Spec
{
    /** @var Spec[] */
    private $children;

    /**
     * @param Spec ...$children
     */
    public function __construct(...$children)
    {
        $this->children = $children;
    }

    /**
     * @inheritDoc
     */
    public function match(QueryBuilder $qb, string $dqlAlias)
    {
        $matches = \array_filter( // disregard empty ones
            \array_map( // map each spec to an Expr and combine them using andX into single Expr
                function (Spec $specification) use ($qb, $dqlAlias) {
                    $match = $specification->match($qb, $dqlAlias);
                    \assert(\is_object($match) && \in_array(\get_class($match), [
                        Expr\Andx::class,
                        Expr\Comparison::class,
                        Expr\Func::class,
                        Expr\Orx::class,
                    ], true), 'Invalid child for OrX expression.');

                    return $match;
                },
                $this->children
            )
        );

        if (empty($matches)) {
            return null;
        }

        return $qb->expr()->andX(...$matches);
    }

    /**
     * @inheritDoc
     */
    public function modifyQuery(AbstractQuery $query): void
    {
        foreach ($this->children as $child) {
            $child->modifyQuery($query);
        }
    }

    /**
     * @inheritDoc
     */
    public function supports(string $className): bool
    {
        foreach ($this->children as $child) {
            if (!$child->supports($className)) {
                return false;
            }
        }

        return true;
    }
}
