<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\OrmExtras\PHPStan;

use Doctrine\ORM\EntityRepository;
use PhpParser\Node\Expr\MethodCall;
use PHPStan\Analyser\Scope;
use PHPStan\Reflection\MethodReflection;
use PHPStan\Type\ArrayType;
use PHPStan\Type\DynamicMethodReturnTypeExtension;
use PHPStan\Type\IntegerType;
use PHPStan\Type\ObjectWithoutClassType;
use PHPStan\Type\Type;
use PHPStan\Type\TypeCombinator;

/**
 * @SuppressWarnings(PHPMD.StaticAccess)
 */
class EntitySpecRepoReturnTypeExtension implements DynamicMethodReturnTypeExtension
{
    /**
     * @inheritDoc
     */
    public function getClass(): string
    {
        return EntityRepository::class;
    }

    /**
     * @inheritDoc
     */
    public function isMethodSupported(MethodReflection $methodReflection): bool
    {
        return 'matchSpec' === $methodReflection->getName() || 'matchSingleSpec' === $methodReflection->getName();
    }

    /**
     * @inheritDoc
     */
    public function getTypeFromMethodCall(
        MethodReflection $methodReflection,
        MethodCall $methodCall,
        Scope $scope
    ): Type {
        $valueType = $this->getValueType($methodReflection);
        if ('matchSingleSpec' === $methodReflection->getName()) {
            return TypeCombinator::addNull($valueType);
        }

        return new ArrayType(new IntegerType(), $valueType);
    }

    public function getValueType(MethodReflection $methodReflection): Type
    {
        $declaring = $methodReflection->getDeclaringClass();
        $ancestor  = $declaring->getAncestorWithClassName(\Doctrine\Persistence\ObjectRepository::class);
        if (!$ancestor) {
            return new ObjectWithoutClassType();
        }

        $typeMap = $ancestor->getActiveTemplateTypeMap();
        $type    = $typeMap->getType('TEntityClass');

        if (!$type) {
            return new ObjectWithoutClassType();
        }

        return $type;
    }
}
