<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\Supplier;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\Config\ConfigCache;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Contracts\Service\Attribute\Required;
use UnexpectedValueException;

/**
 * @template T of object
 */
class UserLoader
{
    private $cachePath;

    private $debug;

    /** @var ManagerRegistry */
    private $doctrine;

    /** @var bool */
    private $loaded = false;

    /** @var null|array<mixed> */
    private $fullUserId;

    /**
     * @param SupplierConfiguration<T> $config
     */
    public function __construct(KernelInterface $kernel, private SupplierConfiguration $config)
    {
        $this->cachePath = $kernel->getCacheDir() . '/cyber/orm_extras.php';
        $this->debug     = $kernel->isDebug();
    }

    #[Required]
    public function setDoctrine(ManagerRegistry $doctrine): void
    {
        $this->doctrine = $doctrine;
    }

    /**
     * @throws \Doctrine\ORM\Exception\ORMException
     *
     * @return null|T
     */
    public function getDefaultUser()
    {
        $this->load();

        if (null === $this->fullUserId) {
            return null;
        }

        $manager = $this->doctrine->getManagerForClass($this->config->userEntityClass);

        if (null === $manager) {
            throw new UnexpectedValueException(\sprintf(
                'Got null as a manager for entity %s',
                $this->config->userEntityClass
            ));
        }

        if ($manager instanceof EntityManagerInterface) {
            return $manager->getReference($this->config->userEntityClass, $this->fullUserId);
        }

        return $manager->find($this->config->userEntityClass, $this->fullUserId);
    }

    /**
     * @SuppressWarnings(PHPMD.ElseExpression)
     */
    public function load(): void
    {
        if ($this->loaded) {
            return;
        }

        $cache     = new ConfigCache($this->cachePath, $this->debug);
        $cacheData = $cache->isFresh() ? \file_get_contents($this->cachePath) : false;

        if (false === $cacheData) {
            $this->fullUserId = $this->fetchUserEntityId();
            $code             = \serialize($this->fullUserId);

            $cache->write($code);
        } else {
            /** @var array<mixed> $unserialized */
            $unserialized     = \unserialize($cacheData, ['allowed_classes' => $this->config->allowedClasses]);
            $this->fullUserId = $unserialized;
        }
        $this->loaded = true;
    }

    /**
     * @return null|array<mixed>
     */
    private function fetchUserEntityId(): ?array
    {
        if (null === $this->config->userEntityClass || null === $this->config->defaultId) {
            //no configuration specified, assuming no default user is needed
            return null;
        }

        $defaultUser = $this->doctrine->getRepository($this->config->userEntityClass)
            ->findOneBy([$this->config->idField => $this->config->defaultId]);
        if (!$defaultUser) {
            return null;
        }

        $metadata = $this->doctrine->getManager()->getClassMetadata($this->config->userEntityClass);

        return $metadata->getIdentifierValues($defaultUser);
    }

    /**
     * @return bool
     */
    public function isLoaded(): bool
    {
        return $this->loaded;
    }
}
