<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\OrmExtras\Functions;

use Doctrine\DBAL\Platforms\MySQLPlatform;
use Doctrine\ORM\Query\AST\Functions\FunctionNode;
use Doctrine\ORM\Query\AST\Node;
use Doctrine\ORM\Query\AST\OrderByClause;
use Doctrine\ORM\Query\AST\PathExpression;
use Doctrine\ORM\Query\Lexer;
use Doctrine\ORM\Query\Parser;
use Doctrine\ORM\Query\SqlWalker;
use Doctrine\ORM\Query\TokenType;

class StringAgg extends FunctionNode
{
    private ?OrderByClause $orderBy = null;

    private PathExpression $expression;

    private Node $delimiter;

    private bool $isDistinct = false;

    public function getSql(SqlWalker $sqlWalker): string
    {
        $platform = $sqlWalker->getConnection()->getDatabasePlatform();
        if ($platform instanceof MySQLPlatform) {
            return \sprintf(
                'group_concat(%s%s %sSEPARATOR %s)',
                ($this->isDistinct ? 'DISTINCT ' : ''),
                $sqlWalker->walkPathExpression($this->expression),
                ($this->orderBy ? $sqlWalker->walkOrderByClause($this->orderBy) . ' ' : ''),
                $sqlWalker->walkStringPrimary($this->delimiter)
            );
        }

        return \sprintf(
            'string_agg(%s%s, %s%s)',
            ($this->isDistinct ? 'DISTINCT ' : ''),
            $sqlWalker->walkPathExpression($this->expression),
            $sqlWalker->walkStringPrimary($this->delimiter),
            ($this->orderBy ? $sqlWalker->walkOrderByClause($this->orderBy) : '')
        );
    }

    public function parse(Parser $parser): void
    {
        // ORM v3 moved constants to TokenType
        $constRef = \enum_exists(TokenType::class) ? TokenType::class : Lexer::class;

        $parser->match($constRef::T_IDENTIFIER);
        $parser->match($constRef::T_OPEN_PARENTHESIS);

        $lexer = $parser->getLexer();
        if ($lexer->isNextToken($constRef::T_DISTINCT)) {
            $parser->match($constRef::T_DISTINCT);

            $this->isDistinct = true;
        }

        $this->expression = $parser->PathExpression(PathExpression::TYPE_STATE_FIELD);
        $parser->match($constRef::T_COMMA);
        $this->delimiter = $parser->StringPrimary();

        if ($lexer->isNextToken($constRef::T_ORDER)) {
            $this->orderBy = $parser->OrderByClause();
        }

        $parser->match($constRef::T_CLOSE_PARENTHESIS);
    }
}
