<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\OrmExtras\Functions;

use Doctrine\ORM\Query\AST;
use Doctrine\ORM\Query\AST\Functions\FunctionNode;
use Doctrine\ORM\Query\AST\PathExpression;
use Doctrine\ORM\Query\Lexer;
use Doctrine\ORM\Query\Parser;
use Doctrine\ORM\Query\QueryException;
use Doctrine\ORM\Query\SqlWalker;
use Doctrine\ORM\Query\TokenType;

/**
 * Allows to use full text search in MySQL
 * - IN NATURAL LANGUAGE MODE
 * - IN NATURAL LANGUAGE MODE WITH QUERY EXPANSION
 * - IN BOOLEAN MODE
 * - WITH QUERY EXPANSION.
 */
class MatchAgainst extends FunctionNode
{
    /** @var array<PathExpression> list of \Doctrine\ORM\Query\AST\PathExpression */
    protected $pathExp;

    /** @var AST\Node|string */
    protected $against;

    /** @var bool */
    protected $booleanMode = false;

    /** @var bool */
    protected $queryExpansion = false;

    /**
     * @param Parser $parser
     *
     * @throws QueryException
     */
    public function parse(Parser $parser): void
    {
        // ORM v3 moved constants to TokenType
        $constRef = \enum_exists(TokenType::class) ? TokenType::class : Lexer::class;
        // match
        $parser->match($constRef::T_IDENTIFIER);
        $parser->match($constRef::T_OPEN_PARENTHESIS);
        // first Path Expression is mandatory
        $this->pathExp   = [];
        $this->pathExp[] = $parser->StateFieldPathExpression();
        // Subsequent Path Expressions are optional
        $lexer = $parser->getLexer();
        while ($lexer->isNextToken($constRef::T_COMMA)) {
            $parser->match($constRef::T_COMMA);
            $this->pathExp[] = $parser->StateFieldPathExpression();
        }
        $parser->match($constRef::T_CLOSE_PARENTHESIS);
        $nextValue = $lexer->lookahead->value ?? '';
        if ('against' !== \mb_strtolower((string) $nextValue)) {
            $parser->syntaxError('against');
        }
        $parser->match($constRef::T_IDENTIFIER);
        $parser->match($constRef::T_OPEN_PARENTHESIS);
        $this->against = $parser->StringPrimary();
        $nextValue     = $lexer->lookahead->value ?? '';
        if ('boolean' === \mb_strtolower((string) $nextValue)) {
            $parser->match($constRef::T_IDENTIFIER);
            $this->booleanMode = true;
        }
        $nextValue = $lexer->lookahead->value ?? '';
        if ('expand' === \mb_strtolower((string) $nextValue)) {
            $parser->match($constRef::T_IDENTIFIER);
            $this->queryExpansion = true;
        }
        $parser->match($constRef::T_CLOSE_PARENTHESIS);
    }

    public function getSql(SqlWalker $walker): string
    {
        $fields = [];
        foreach ($this->pathExp as $pathExp) {
            $fields[] = $pathExp->dispatch($walker);
        }
        $against = $walker->walkStringPrimary($this->against)
            . ($this->booleanMode ? ' IN BOOLEAN MODE' : '')
            . ($this->queryExpansion ? ' WITH QUERY EXPANSION' : '');

        return \sprintf('MATCH (%s) AGAINST (%s)', \implode(', ', $fields), $against);
    }
}
