Query specs allow you to make query logic simpler and more reusable. This features was inspired by article
[On Taming Repository Classes][repo-article].

# Overview

The bundle provides some of the core *Specs* and the backbone for easily writing your own application specific ones. As
the article outlines this can also simplify the testability of your repositories and "avoid the problem of combinatorial
explosion of test-cases".

> All specs provided by this feature are thorougly tested. Each spec you create on your project should be tested in
> isolation so that repo tests can be safely mocked.

# Using Specs

To start using specs simply make your repository class `use EntitySpecRepoTrait`. It will provided the
`matchSpec(Spec $spec)` function which will your main entry point for all spec queries.

After you have setup your specifications or just want to use the built in ones, get the database results like so:

```php
<?php
use App\YourCustomSpec;
use App\YourRepository;
use Cyber\OrmExtras\Spec;

$spec = new Spec\OnlyPage(1, new Spec\AndX(
    new Spec\Equals('last', 'john'),
    new Spec\Equals('first', 'smith'),
    new YourCustomSpec()
));

/** @var Spec\EntitySpecRepoTrait $repo */
$repo = new YourRepository();
$results = $repo->matchSpec($spec);

```

> If you need some custom logic for spec execution feel free to implement `matchSpec()` function on your own by
> following the original as a template.

## Default Alias

The internal query builder for `matchSpec()` uses ***spec*** as the default entity alias. If you wish to define custom
alias simply add **defaultAlias** property to your repository class that uses `EntitySpecRepoTrait` like so
`protected $defaultAlias = 'myAlias'`. The value of this property will be used as the default alias for that repository.

# Included Specs

Below is the list of specs that are provided in this bundle.

## AndX

The `Cyber\OrmExtras\Spec\AndX` spec corresponds to an **AND** expression. It can take 1+ arguments of other `Spec`s and
will join them in the query as **AND**.

Example: `$spec = new AndX(new SomeSpec(), new SomeOtherSpec(), ...)`

## Any

The `Cyber\OrmExtras\Spec\Any` spec corresponds to an empty where expression. It will result in mathching everything.

Example: `$spec = new Any()`

## OrX

Same as `Cyber\OrmExtras\Spec\AndX` but joins the provided specs with an **OR**.

Example: `$spec = new OrX(new SomeSpec(), new SomeOtherSpec(), ...)`

## AsArray

The `Cyber\OrmExtras\Spec\AsArray` spec simply changes the hydration mode to be an *array* instead of the default
*object*.

Example: `$spec = new AsArray(new AndX(...))`

## Equals

The `Cyber\OrmExtras\Spec\Equals` spec provides an ability for matching a single field to a literal value.

Example: `$spec = new Equals('first', 'john')`

## Regex

The `Cyber\OrmExtras\Spec\Regex` spec provides an ability for matching a regex expression.

Example: `$spec = new Regex('field', 'reg.*ex')`

> This will only work if you have registered the [Regex Function](regex-function.md) as `regex`.

## OnlyPage

The `Cyber\OrmExtras\Spec\OnlyPage` spec allows you to limit the returned results to a particular page. It takes the
page number, an optional child `Spec` and an optional records per page argument.

Example: `$spec = new OnlyPage(1, new Equals('first', 'john'), 5)`

> see the default records per page value in the constructor.

# Writing Your Specs

To write your spec simply create a class which implements `Cyber\OrmExtras\Spec`. Once all interface functions are
implemented you can use this spec anywhere in your application. Below are brief discussion of each function.

## supports

The `supports` function receives the class name of the repository's entity that is going to be matched. It needs to
return `true` or `false` identifying whether such class is supported by this spec.

## match

The `match` function receives 2 arguments `QueryBuilder $qb` and `string $dqlAlias`. Your spec can use these values to
update the query builder any way necessary for this spec.

In the end it must return an expression that would be acceptable by `$qb->where($expression)`.

The `match` can return **NULL** for specs that do not result in any new *WHERE* conditions (such as joining tables,
setting result limits, etc).

> If your spec does not need to do any filtering and if it has child specs as argument
> then most likely you want to return the result of `$child->match($qb, $dqlAlias)`

## modifyQuery

The `modifyQuery` receives the `Query $query` generated by the `QueryBuilder` after `match()` function completes. Here
you can add extra modification to the query to change its behavior. For example:

* Modify cache behavior
* Set query hints
* Limit results (OnlyPages spec does this)

Nothing is expected as a return from this function.

## Helpful Base Classes

Below are a few abstract Spec classes you can extend from when using common features.

### Cyber\OrmExtras\Spec\ChildSpec

Often you will need a `Spec` which can accept another `Spec` as a child. This base class provides a constructor to
accept such child as argument, and default implementations of each of the abstract functions of `Spec` which directly
proxy the calls to the `child`. Furthermore, child can be optional.

This way if your spec only needs to do something in `modifyQuery()`. You simply override only that method, and don't
need to worry about implementing the rest.

> don't forget to call the parent implementation of methods you override, unless you provided your own
> implementation and do not need the default behavior.

### Cyber\OrmExtras\Spec\JoiningSpec

Another frequently used `Spec` is one that joins another table. This base class is tailed for implement such `Spec`s.

It extends from `ChildSpec`, so by default is able to accept an optional child `Spec`. However, this class does not
allow you to override the default `match` and `supports` functions as they have implemented a very specific logic for
managing the joined table and child spec.

Instead, you have to provide implementation/values for the following:

* `$joinAlias`: should be the alias of your join. This will be passed as new alias to child spec's `match()` call
* `$joinClass`: should be the class name of your join. This will be passed to child spec's `supports()` call
* `joinMatch()`: this is where your `Spec` should implement the logic that usually goes into `match()` function
* `joinSupports()`: this is where your `Spec` should implement the logic that usually goes into `supports()` function

# PHPStan Integration

This bundle provides a PHPStan plugin to assist with properly typing the return value of `matchSpec()` calls for
repositories implementing the `EntitySpecRepoTrait`.

To activate the plugin add the following to your PHPStan config file:

```neon
services:
    - class: Cyber\OrmExtras\PHPStan\EntitySpecRepoReturnTypeExtension
      tags: [phpstan.broker.dynamicMethodReturnTypeExtension]
```

[repo-article]: https://beberlei.de/2013/03/04/doctrine_repositories.html
