<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\UploadBundle\Component;

use Cyber\UploadBundle\Component\HandlerInterface;
use Cyber\UploadBundle\Component\UploadHolder;
use Cyber\UploadBundle\DependencyInjection\CyberUploadExtension;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Tests\Cyber\UploadBundle\Entity\MockUpload;
use Tests\Cyber\UploadBundle\Mock\DiInjector;

/**
 * @group functional
 */
class AwsHandlerFunctionalTest extends TestCase
{
    /** @var string */
    private static $tempFile;

    /** @var array<mixed> */
    private static $configs;

    /** @var HandlerInterface */
    private $handler;

    /** @var UploadedFile */
    private $uploadedFile;

    /** @var string */
    private static $dummyText = 'just some random data';

    public static function setUpBeforeClass(): void
    {
        //cyber_upload: config
        self::$configs = [
            'handler'         => 'aws',
            'naming_strategy' => 'entity_id',
            'aws'             => [
                'bucket' => 'bydevs',
                'path'   => 'testing-scripts/CyberUploadBundle',
                'key'    => 'AKIAJ642UMTRWWHXHSOA',
                'secret' => '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',
            ],
            'mapping' => [
                ['class' => 'Cyber\UploadBundle\Tests\Entity\MockUpload', 'path' => 'MockUpload'],
            ],
        ];

        $file = \tempnam(\sys_get_temp_dir(), 'S3_');
        self::assertInternalType('string', $file);
        self::$tempFile = $file;
        \file_put_contents(self::$tempFile, self::$dummyText);
    }

    public static function tearDownAfterClass(): void
    {
        \chmod(self::$tempFile, 0777);
    }

    protected function setUp(): void
    {
        $extension = new CyberUploadExtension();

        $container = new ContainerBuilder();
        $container->registerExtension($extension);

        $container->loadFromExtension($extension->getAlias(), self::$configs);
        $container->setDefinition(
            'Tests\Cyber\UploadBundle\Mock\DiInjector',
            (new Definition('Tests\Cyber\UploadBundle\Mock\DiInjector'))->setPublic(true)->setAutowired(true)
        );
        $container->compile();

        /** @var DiInjector $diInjector */
        $diInjector         = $container->get('Tests\Cyber\UploadBundle\Mock\DiInjector');
        $this->handler      = $diInjector->getCyberUpload();
        $this->uploadedFile = new UploadedFile(self::$tempFile, 'original.txt');
    }

    public function testUpload(): MockUpload
    {
        $entity = new MockUpload(\random_int(1000, 9999));

        $entity->setFile($this->uploadedFile);

        $this->handler->upload($entity);

        //as long as we don't get exception we are good. add assertion to prevent no assert failures
        $this->assertTrue(true);

        return $entity;
    }

    /**
     * @depends testUpload
     *
     * @param UploadHolder $entity
     */
    public function testDownload($entity): void
    {
        $url = $this->handler->getURL($entity, ['signed' => true]);

        $curl = \curl_init($url);
        $this->assertInternalType('resource', $curl);

        \curl_setopt($curl, CURLOPT_HEADER, true);
        \curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        $result = \curl_exec($curl);
        $this->assertInternalType('string', $result);

        $headers = $this->getHeadersFromCurlResponse($result);
        \curl_close($curl);

        $this->assertTrue(false !== \mb_strpos($headers['Content-Disposition'], 'theFirstFile.txt'));
    }

    /**
     * @depends testUpload
     *
     * @param UploadHolder $entity
     */
    public function testPublicUrl($entity): void
    {
        $url = $this->handler->getURL($entity);
        $this->assertNotContains('X-Amz-Credential', $url);
    }

    /**
     * @depends testUpload
     *
     * @param UploadHolder $entity
     * @SuppressWarnings(PHPMD.LongVariable)
     */
    public function testCopy($entity): void
    {
        $destinationEntity = new MockUpload(2);

        $this->handler->copy($entity, $destinationEntity);

        $destinationEntityUrlAfterCopy = $this->handler->getURL($destinationEntity);

        $this->assertEquals(
            self::$dummyText,
            \file_get_contents($destinationEntityUrlAfterCopy)
        );
        $this->assertGreaterThan(
            0,
            \mb_strpos($destinationEntityUrlAfterCopy, '2.txt'),
            'There are uploaded file on entity 2 or it has not proper name and extension.'
        );
    }

    /**
     * @depends testUpload
     *
     * @param UploadHolder $entity
     */
    public function testDelete($entity): void
    {
        $this->handler->delete($entity);

        //as long as we don't get exception we are good. add assertion to prevent no assert failures
        $this->assertTrue(true);
    }

    /**
     * @param string $response
     *
     * @return string[]
     * @SuppressWarnings(PHPMD.ElseExpression)
     */
    private function getHeadersFromCurlResponse($response): array
    {
        $headers  = [];
        $boundary = \mb_strpos($response, "\r\n\r\n");
        if (false === $boundary) {
            $boundary = null;
        }

        $headerText = \mb_substr($response, 0, $boundary);

        foreach (\explode("\r\n", $headerText) as $i => $line) {
            if (0 === $i) {
                $headers['http_code'] = $line;
            } else {
                [$key, $value] = \explode(': ', $line);

                $headers[$key] = $value;
            }
        }

        return $headers;
    }
}
