<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\UploadBundle\Component;

use Symfony\Component\HttpFoundation\File\UploadedFile;

interface UploadHolder
{
    /**
     * Returns unique slug for each entity.
     *
     * This is typically generated based off an ID. Can be an integer for numeric ids or string for uuid entities.
     *
     * If string must be at least 3 characters long. Should only contain characters that can be part of file path
     * for the used handler.
     *
     * If null is returned the data will not be uploaded as the path cannot be calculated.
     *
     * @return null|int|string
     */
    public function getStorageSlug();

    /**
     * Return file that was uploaded by user.
     *
     * @return null|UploadedFile
     */
    public function getFile(): ?UploadedFile;

    /**
     * Return file extension without the dot.
     *
     * @return string
     */
    public function getExt(): string;

    /**
     * If this function returns true during upload, the uploaded file will be downloadable to anyone with a link
     * without need for creating a signed request.
     *
     * @return bool true for files which should be publicly accessible
     *
     * @SuppressWarnings(PHPMD.BooleanGetMethodName)
     */
    public function getPublic(): bool;

    /**
     * Returns the filename.
     *
     * Used to set the name of the file that will encoded in the url for download. This only works on private files
     * in some handlers.
     *
     * If using "file_name" naming_strategy you MUST NOT change the name once the entity has been persisted. Otherwise
     * the generate link will be invalid. To rename the file you should create a new entity with the new name, then use
     * the copy() function of the handler, then delete the old entity.
     *
     * NOTE: You don't need to download the file from storage and save it in the new entity. The handler handles the
     * copy on the storage side directly.
     *
     * @return string
     */
    public function getName(): string;
}
