<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\UploadBundle\DependencyInjection;

use Cyber\UploadBundle\Component\Handler\AwsHandler;
use Cyber\UploadBundle\Component\Handler\LocalStorageHandler;
use Cyber\UploadBundle\Component\Handler\NoOpHandler;
use Cyber\UploadBundle\Component\HandlerInterface;
use LogicException;
use RuntimeException;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

/**
 * This is the class that loads and manages your bundle configuration.
 *
 * @see http://symfony.com/doc/current/cookbook/bundles/extension.html
 */
class CyberUploadExtension extends Extension
{
    /**
     * @inheritdoc
     */
    public function load(array $configs, ContainerBuilder $container): void
    {
        $loader = new YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/config'));
        $loader->load('services.yml');

        $configuration = new Configuration();
        $config        = $this->processConfiguration($configuration, $configs);

        $this->setupHandlerService($config, $container, $loader);

        if ($config['mapping']) {
            $mapping = [];

            foreach ($config['mapping'] as $key => $map) {
                $class      = $map['class'];
                $path       = \trim(\str_replace('\\', '/', $map['path']), '/');
                $configPath = $this->getAlias() . '.mapping.' . $key;

                if (\array_key_exists($class, $mapping)) {
                    $ex = new LogicException('Duplicate entry exists for class: ' . $class . ' at ' . $configPath);
                    throw $ex;
                }

                if (\in_array($path, $mapping, true)) {
                    $ex = new LogicException('Duplicate entry exists for path: ' . $path . ' at ' . $configPath);
                    throw $ex;
                }

                $mapping[$class] = $path;
            }
            $container->setParameter('cyber.upload.mapping', $mapping);
        }
    }

    private function setupHandlerService(array $config, ContainerBuilder $container, YamlFileLoader $loader): void
    {
        // for service just setup alias
        if ('service' === $config['handler']) {
            $container->setAlias(HandlerInterface::class, $config['service']);

            return;
        }

        // otherwise do more configurations
        switch ($config['handler']) {
            case 'aws':
                $loader->load('aws-handler.yml');
                $class = AwsHandler::class;
                $arg   = [
                    '$s3Client'       => new Reference('cyber.upload.s3client'),
                    '$bucket'         => $config['aws']['bucket'],
                    '$path'           => $config['aws']['path'],
                    '$namingStrategy' => $config['naming_strategy'],
                ];
                $container->setParameter('cyber.upload.key', $config['aws']['key']);
                $container->setParameter('cyber.upload.secret', $config['aws']['secret']);
                break;
            case 'local':
                $class = LocalStorageHandler::class;
                $arg   = [
                    '$storagePath' => $config['local']['path'],
                ];
                break;
            case 'noop':
                $class = NoOpHandler::class;
                $arg   = [];
                break;
            default:
                throw new RuntimeException('Unsupported handler specified');
        }

        $container->setDefinition(
            HandlerInterface::class,
            (new Definition($class))
                ->setArguments($arg)
                ->setAutowired(true)
                ->setLazy(true) //need to be lazy cause we inject into twig extension, only helps is lazy supported
        );
    }
}
