<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\UploadBundle\DependencyInjection;

use Cyber\UploadBundle\Component\Handler\AwsHandler;
use Cyber\UploadBundle\Component\Handler\LocalStorageHandler;
use Cyber\UploadBundle\Component\Handler\NoOpHandler;
use Cyber\UploadBundle\DependencyInjection\CyberUploadExtension;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Tests\Cyber\UploadBundle\Mock\DiInjector;

class CyberUploadExtensionTest extends TestCase
{
    /** @var ContainerBuilder */
    private $container;

    /** @var CyberUploadExtension */
    private $extension;

    /**
     * @var array<string, mixed>
     */
    private $defaultConfig = [
        'handler'         => 'aws',
        'aws'             => [
            'bucket' => 'a',
            'path'   => 'b',
            'key'    => 'c',
            'secret' => 'd',
        ],
        'mapping'         => [],
        'naming_strategy' => 'entity_id',
    ];

    public function testLoad(): void
    {
        $configs = $this->defaultConfig;

        $configs['mapping'] = [
            ['class' => 'Some\Class1', 'path' => 'Some/Path'],
            ['class' => 'Some\Class2', 'path' => '/Some\Path2\\'],
        ];

        $this->container->loadFromExtension($this->extension->getAlias(), $configs);
        $this->container->setDefinition(
            'Tests\Cyber\UploadBundle\Mock\DiInjector',
            (new Definition('Tests\Cyber\UploadBundle\Mock\DiInjector'))
                ->setPublic(true)->setAutowired(true)
        );
        $this->container->compile();

        $this->assertTrue($this->container->has('Tests\Cyber\UploadBundle\Mock\DiInjector'));
        /** @var DiInjector $diInjector */
        $diInjector = $this->container->get('Tests\Cyber\UploadBundle\Mock\DiInjector');
        $handler    = $diInjector->getCyberUpload();
        $this->assertNotNull($handler);
        $this->assertInstanceOf(AwsHandler::class, $handler);
        $parsedMapping = $this->container->getParameter('cyber.upload.mapping');
        $this->assertNotNull($parsedMapping);

        $this->assertArrayHasKey('Some\Class1', $parsedMapping);
        $this->assertArrayHasKey('Some\Class2', $parsedMapping);

        $this->assertEquals('Some/Path', $parsedMapping['Some\Class1']);
        $this->assertEquals('Some/Path2', $parsedMapping['Some\Class2']); //testing trimming of path
    }

    public function testLocalLoad(): void
    {
        $configs = [
            'handler' => 'local',
            'local'   => [
                'path' => '/tmp',
            ],
        ];

        $this->doHandlerTest($configs, LocalStorageHandler::class);
    }

    public function testNoOpLocad(): void
    {
        $configs = [
            'handler' => 'noop',
        ];

        $this->doHandlerTest($configs, NoOpHandler::class);
    }

    /**
     * @group integration
     */
    public function testYmlLoad(): void
    {
        $loader = new YamlFileLoader($this->container, new FileLocator(__DIR__));
        $loader->load('sample_config.yml');
        $this->container->compile();

        $this->assertTrue($this->container->has('Tests\Cyber\UploadBundle\Mock\DiInjector'));
        /** @var DiInjector $diInjector */
        $diInjector = $this->container->get('Tests\Cyber\UploadBundle\Mock\DiInjector');
        $this->assertNotNull($diInjector->getCyberUpload());
        $parsedMapping = $this->container->getParameter('cyber.upload.mapping');
        $this->assertNotNull($parsedMapping);

        $this->assertArrayHasKey('Company\Bundle\MainUpload', $parsedMapping);
        $this->assertArrayHasKey('Company\Bundle\UploadPreview', $parsedMapping);

        $this->assertEquals('main/uploads', $parsedMapping['Company\Bundle\MainUpload']);
        $this->assertEquals('previews', $parsedMapping['Company\Bundle\UploadPreview']); //testing trimming of path
    }

    public function testLoadFail(): void
    {
        $this->expectException('\Symfony\Component\Config\Definition\Exception\InvalidConfigurationException');
        $this->expectExceptionMessage('The value "zzz" is not allowed for path');

        $configs = $this->defaultConfig;

        $configs['handler'] = 'zzz';

        $this->container->loadFromExtension($this->extension->getAlias(), $configs);
        $this->container->compile();

        $this->assertTrue($this->container->has('cyber.upload.handler'));
    }

    public function testDuplicateClass(): void
    {
        $this->expectException('\LogicException');
        $this->expectExceptionMessage('Duplicate entry exists for class');

        $configs = $this->defaultConfig;

        $configs['mapping'] = [
            ['class' => 'Some\Class', 'path' => 'Some\Path'],
            ['class' => 'Some\Class', 'path' => 'Some\Path2'],
        ];

        $this->container->loadFromExtension($this->extension->getAlias(), $configs);
        $this->container->compile();
    }

    public function testDuplicatePath(): void
    {
        $this->expectException('\LogicException');
        $this->expectExceptionMessage('Duplicate entry exists for path');

        $configs = $this->defaultConfig;

        $configs['mapping'] = [
            ['class' => 'Some\Class', 'path' => 'Some\Path'],
            ['class' => 'Some\Class2', 'path' => 'Some\Path'],
        ];

        $this->container->loadFromExtension($this->extension->getAlias(), $configs);
        $this->container->compile();
    }

    public function testNamingStrategy(): void
    {
        $this->expectException('\Symfony\Component\Config\Definition\Exception\InvalidConfigurationException');
        $this->expectExceptionMessage('cyber_upload.naming_strategy');

        $configs                    = $this->defaultConfig;
        $configs['naming_strategy'] = null;

        $this->container->loadFromExtension($this->extension->getAlias(), $configs);
        $this->container->compile();
    }

    protected function setUp(): void
    {
        $this->extension = new CyberUploadExtension();
        $this->container = new ContainerBuilder();
        $eventDispatcher = self::getMockBuilder(EventDispatcherInterface::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->container->set('event_dispatcher', $eventDispatcher);
        $this->container->registerExtension($this->extension);
    }

    /**
     * @param array<mixed> $configs
     * @param class-string $expectedClass
     */
    private function doHandlerTest(array $configs, string $expectedClass): void
    {
        $this->container->loadFromExtension($this->extension->getAlias(), $configs);
        $this->container->setDefinition(
            'Tests\Cyber\UploadBundle\Mock\DiInjector',
            (new Definition('Tests\Cyber\UploadBundle\Mock\DiInjector'))
                ->setPublic(true)->setAutowired(true)
        );
        $this->container->compile();

        $this->assertTrue($this->container->has('Tests\Cyber\UploadBundle\Mock\DiInjector'));
        /** @var DiInjector $diInjector */
        $diInjector = $this->container->get('Tests\Cyber\UploadBundle\Mock\DiInjector');
        $handler    = $diInjector->getCyberUpload();
        $this->assertNotNull($handler);
        $this->assertInstanceOf($expectedClass, $handler);
    }
}
