# About
To provide simple integration of Doctrine Entities and Amazon S3 for file upload and storage.

# Configuration

This bundle uses the following configuration structure. You should configure all options as needed by your project.

```yaml
cyber_upload:
    handler: "aws" # stands for Amazon Web Services for now will be the only supported handler
    aws:    # when handler is 'aws' these parameters are should be required, non should be empty
        bucket: ~   # the bucket where uploads will be stored
        path: ~     # the path inside the bucket where uploads will be stored (ex. 'company/uploads')
        key: ~      # the key used for authentication with S3
        secret: ~   # the secret used for authentication with S3
    naming_strategy: entity_id # One of "entity_id"; "file_name"
    mapping: # (optional, array prototype) custom path mapping for each entity, Default: ~
        class: # Fully qualified class name to map to a path below
        path:  # The storage path the above class is mapped to
```

## Sample Configuration
```yaml
cyber_upload:
    handler: "aws" 
    aws: 
        bucket: my-bucket
        path: uploads
        key: public-key
        secret: secret
    naming_strategy: file_name
    mapping: 
        - { class: "Company\Bundle\MainUpload", path: "main/uploads" }
```

## Handlers
This bundle supports 3 handlers:
* aws - Amazon's S3 service
* local - local file storage
* noop - a no-operation handler, used for testing
* service - your custom service

After choosing the handler you need to provide the configuration values corresponding to the handler.

## Storage Path
The storage path is calculated based on the **bucket**, **path**, **class**, and **id** of entity.

Full path will is: `[aws.bucket]/[aws.path]/[class mapping]/[group]/[name](/[file_name]).[extension]`

The **[class mapping]** is calculated based on FQCN. If the class name exists in the **[mapping]** section the path
corresponding to that class will be use. Otherwise `md5()` hash of the FQCN will be calculated and used.

The **[group]** is calculated based on `getStorageSlug()` of entity:
* In case it is integer then, by dividing it by **10,000** and rounding down *+1*.
* In case it it string then, by taking first 2 characters of the string

The **[name]** is calculated based on the `getStorageSlug()` as well:
* In case of integer it is the whole integer
* In case of string it is the remaining characters after removing the first 2 from the front.

The **[file_name]** will be present if `file_name` *naming_strategy* is used in the configuration and it will be
the actual name of the uploaded file with all the illegal characters escaped.

# Usage

To use this bundle simply implement `UploadHolder` on your entity responsible for storing data about the file.

Then when user uploads the file via a form, set the corresponding fields on your entity and persist it. When 
`UploadBundle` detects an entity being persisted it will upload the file returned by the `getFile()` function to its
destination. The name of the file at destination will be combination of value returned by `getId()` (the id of the 
entity) and `getExt()`. Further more, files will be placed into a sub directories starting with `1` up to 10,000 files 
in each directory.

## Naming Strategy
The original naming strategy was to use the entity ID. This worked create with private uploads as you could override the
name when signing the url for download. However, for any public files the user would download an arbitrary numeric value
file (as it is impossible to override the download name on public urls), which is often inconvenient. 

The `file_name` naming strategy stores the files in the storage with their original names. This way, when public files
are downloaded they will have proper filenames. This however comes with its own drawback. WHEN USING THIS STRATEGY YOU
ARE NOT ALLOWED TO SIMPLY CHANGE THE NAME OF THE FILE WHICH IS STORED IN THE ENTITY, since changing the name would break
the generated URL. So to rename the file you will need to do the following:
1. create entity with new name
2. use the handler's `copy()` operation with old and new entities
3. delete the old entity

> You don't need to download the actual file locally, the handler copies the file in the storage engine directly, all 
> it needs are the source and dest paths, which are generated based on the data of the 2 supplied entities. 

## Sample Entity

Here is a sample entity implementing `UploadHolder`. Fields `id` and `ext` are crucial as they are required
to later fetch the uploaded file from its storage location. Also note that `file` field does not have `Column`
attribute as it does not need to be stored in the database. This field is used to only temporarily store the uploaded
file.

If `file` field is null, our handle will simply skip the upload process, thus allowing you to change any other fields
in the entity and persist it without having user upload the file each time. 

> You should never change the values returned by `getStorageSlug()` or `getExt()` once the file has been uploaded.

```php
<?php
use Cyber\UploadBundle\Component\UploadHolder;
use Symfony\Component\HttpFoundation\File\UploadedFile;

class Upload implements UploadHolder
{
    #[ORM\Column(name: "upl_id", type: "integer")]
    #[ORM\Id]
    #[ORM\GeneratedValue(strategy: "AUTO")]
    private $id;

    /**
     * @var string for storing original filename
     */
     #[ORM\Column(name: "upl_name", type: "string", length: 30, nullable: false)]
    private $name;

    /**
     * @var string extension of the uploaded file
     */
     #[ORM\Column(name: "upl_ext", type: "string", length: 7, nullable: false)]
    private $ext;
    
    /**
     * @var string extension of the uploaded file
     */
     #[ORM\Column(name: "upl_public", type: "boolean")]
    private $public;

    /**
     * @var UploadedFile temporary storage of the UploadedFile object coming from forms
     */
    private $file;
   
    public function getStorageSlug()
    {
        return $this->id;
    }

    public function getId()
    {
        return $this->id;
    }
    
    public function getExt(): string
    {
        return $this->ext;
    }

    public function getFile(): ?UploadedFile
    {
        return $this->file;
    }
    
    public function getPublic(): bool 
    {
        return $this->public;
    }
    
    public function getName(): string
    {
        return $this->name;
    }
        
    public function setFile(UploadedFile $file = null)
    {
        $this->file = $file;

        return $this;
    }
    
    //other setters/getters
    
}
```

## Twig

The bundle also provides a convenient twig function to render the url to the stored file inside a twig template.
The function accept one argument which must be your entity implementing the `UploadHolder`.

```twig
    {{ cyber_upload_url(uploadEntity) }}
```

## Service

If you ever need to get the uploaded file url inside your controller, you can inject the 
`Cyber\UploadBundle\Component\HandlerInterface` service:
 
```php
<?php
    public function __construct(Cyber\UploadBundle\Component\HandlerInterface $handler){
        // ...
        $handler->getURL($uploadEntity);
    }
```

The same service also allows to trigger manual uploads of files if such need ever arises:

```php
<?php
    $handler->getURL($uploadEntity);
```

## Public Files
Services such as Amazon S3 allow to upload file with public access. If `getPublic()` method returns **true** the handler
will upload the file with public flag. 

By default however, when generating URL even for public files, it will calculate a signature and add an expiration date
to that url. This is needed by another mechanism which allows to specify proper filename name during downloads. If you do
not care about the filename (as in cases with inline images for example), then you can choose to get unsigned URL for
public files. 

To generate unsigned url, pass an option `'signed' => false` to the url generating functions (`cyber_upload_url` in twig or 
`getURL` in php). Unsigned URL never expire and saves CPU cycles on signature calculations.
