<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\UploadBundle\Component;

use InvalidArgumentException;

/**
 * A common interface for upload handlers.
 *
 * Implementing classes must ensure not to throw any exceptions other than those specified in the function documentation
 */
interface HandlerInterface
{
    /**
     * uploads the $file from the $uploadEntity to the storage location.
     *
     * @param UploadHolder $uploadEntity
     *
     * @throws InvalidArgumentException if entity does not have an id set
     * @throws HandlerException         if any error occurred during upload operation
     */
    public function upload(UploadHolder $uploadEntity): void;

    /**
     * delete the file of the $uploadEntity in the storage location.
     *
     * @param UploadHolder $uploadEntity the entity whose related file is to be deleted
     *
     * @throws InvalidArgumentException if entity does not have an id set
     * @throws HandlerException         if any error occurred during delete operation
     */
    public function delete(UploadHolder $uploadEntity): void;

    /**
     * Copy file on S3 from $source to $destination.
     *
     * @param UploadHolder $source
     * @param UploadHolder $destination
     */
    public function copy(UploadHolder $source, UploadHolder $destination): void;

    /**
     * Imports the file represented by the URL under the specified destination.
     *
     * @param string       $fromUrl     url to get the file from
     * @param UploadHolder $destination entity to upload under
     */
    public function import(string $fromUrl, UploadHolder $destination): void;

    /**
     * returns the URL by which the stored $file of the $uploadEntity can be reached.
     *
     * @param UploadHolder $uploadEntity
     * @param array<mixed> $options      handler specific options. See doc block above this method in the handler you are
     *                                   using.
     *
     * @throws InvalidArgumentException if entity does not have an id set
     *
     * @return string
     */
    public function getURL(UploadHolder $uploadEntity, array $options = []): string;

    /**
     * Checks whether the stored $file of the $uploadEntity exists on S3.
     *
     * @param UploadHolder $uploadEntity
     *
     * @return bool
     */
    public function exists(UploadHolder $uploadEntity): bool;
}
