<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\UploadBundle\Component\Handler;

use Aws\S3\S3Client;
use Cyber\UploadBundle\Component\HandlerException;
use Cyber\UploadBundle\Component\HandlerInterface;
use Cyber\UploadBundle\Component\PathResolver;
use Cyber\UploadBundle\Component\UploadHolder;
use Exception;
use InvalidArgumentException;
use Symfony\Component\OptionsResolver\OptionsResolver;

/**
 * Handler for storing files in AWS S3 service.
 */
class AwsHandler implements HandlerInterface
{
    private $client;

    private $bucket;

    private $path;

    private $resolver;

    private $namingStrategy;

    public function __construct(
        S3Client $s3Client,
        PathResolver $resolver,
        string $bucket,
        string $path,
        string $namingStrategy
    ) {
        $this->client         = $s3Client;
        $this->resolver       = $resolver;
        $this->bucket         = $bucket;
        $this->path           = $path;
        $this->namingStrategy = $namingStrategy;
    }

    /**
     * Copy uploaded to S3 file from one UploadedEntity to another on S3 only.
     * Check acl's for operation on  UploadedEntity to copy file to.
     *
     * @param UploadHolder $uploadEntitySource
     * @param UploadHolder $uploadEntityDestination
     *
     * @SuppressWarnings(PHPMD.LongVariable)
     */
    public function copy(
        UploadHolder $uploadEntitySource,
        UploadHolder $uploadEntityDestination
    ): void {
        $keySource      = $this->getStorageKey($uploadEntitySource);
        $keyDestination = $this->getStorageKey($uploadEntityDestination);
        $acl            = $uploadEntityDestination->getPublic() ? 'public-read' : 'private';

        try {
            $this->client->copy(
                $this->bucket,
                $keySource,
                $this->bucket,
                $keyDestination,
                $acl
            );
        } catch (Exception $exception) {
            throw new HandlerException(
                'File copy failed.',
                $uploadEntitySource,
                $exception
            );
        }
    }

    /**
     * @inheritDoc
     */
    public function import(string $fromUrl, UploadHolder $destination): void
    {
        $key = $this->getStorageKey($destination);
        $acl = $destination->getPublic() ? 'public-read' : 'private';

        try {
            $this->client->upload($this->bucket, $key, \fopen($fromUrl, 'rb'), $acl);
        } catch (Exception $e) {
            throw new HandlerException('File upload failed.', $destination, $e);
        }
    }

    /**
     * @inheritDoc
     */
    public function upload(UploadHolder $uploadEntity): void
    {
        $file = $uploadEntity->getFile();
        if (!$file) {
            return;
        }

        $pathName = $file->getPathname();

        if (!\file_exists($pathName)) {
            throw new HandlerException('File does not exist at: ' . $pathName, $uploadEntity);
        }

        $this->import($pathName, $uploadEntity);
    }

    /**
     * @inheritDoc
     */
    public function delete(UploadHolder $uploadEntity): void
    {
        $key = $this->getStorageKey($uploadEntity);

        try {
            $this->client->deleteObject([
                'Bucket' => $this->bucket,
                'Key'    => $key,
            ]);
        } catch (Exception $e) {
            throw new HandlerException('File deletion failed', $uploadEntity, $e);
        }
    }

    /**
     * Supported Options:
     * - signed (bool) - generate signed or unsigned url, private uploads must be signed, for any other options to apply
     * it MUST be signed.
     *
     * - attachment (bool) - include a header to force browser download instead of display inline.
     *
     * - content-type (string) - set the ContentType header S3 will respond with for this url.
     *
     * {@inheritDoc}
     */
    public function getURL(UploadHolder $uploadEntity, array $options = []): string
    {
        $optionsResolver = new OptionsResolver();
        $optionsResolver
            ->setDefaults([
                'signed'       => null,
                'attachment'   => false,
                'content-type' => null,
            ])
            ->setAllowedTypes('signed', ['null', 'bool'])
            ->setAllowedTypes('attachment', 'bool')
            ->setAllowedTypes('content-type', ['null', 'string']);
        $options = $optionsResolver->resolve($options);

        $name = $this->getStorageKey($uploadEntity);

        if ($uploadEntity->getPublic() && !$options['signed']) {
            // public urls not requiring signature can be left unsigned
            return $this->client->getObjectUrl($this->bucket, $name);
        }

        if (!$uploadEntity->getPublic() && false === $options['signed']) {
            // if user explicitly specified unsigned but file is private need to at least report a warning
        }

        // all others must be signed
        $s3Args = [
            'Bucket'                     => $this->bucket,
            'Key'                        => $name,
            'ResponseContentDisposition' => ($options['attachment'] ? 'attachment' : 'inline') . '; filename="' .
                $uploadEntity->getName() . '.' . $uploadEntity->getExt() . '"',
        ];
        if (isset($options['content-type'])) {
            $s3Args['ResponseContentType'] = $options['content-type'];
        }

        $cmd     = $this->client->getCommand('GetObject', $s3Args);
        $request = $this->client->createPresignedRequest($cmd, '+10 minutes');

        return (string) $request->getUri();
    }

    /**
     * @inheritDoc
     */
    public function exists(UploadHolder $uploadEntity): bool
    {
        if (null === $uploadEntity->getStorageSlug()) {
            return false;
        }

        $key = $this->getStorageKey($uploadEntity);

        return $this->client->doesObjectExist($this->bucket, $key);
    }

    /**
     * @param UploadHolder $uploadEntity
     *
     * @throws InvalidArgumentException if entity does not have an id set
     *
     * @return string
     */
    private function getStorageKey(UploadHolder $uploadEntity): string
    {
        $pathParts = [];

        if ($this->path) {
            $pathParts[] = $this->path;
        }

        $entityPath = $this->resolver->resolve($uploadEntity);

        if ($entityPath) {
            $pathParts[] = $entityPath;
        }

        $slug = $uploadEntity->getStorageSlug();

        if (null === $slug) {
            throw new HandlerException('Cannot get storage key without a storage slug from the entity.');
        }

        $pathParts[] = $this->parseSlug($slug);
        if ('file_name' === $this->namingStrategy) {
            $pathParts[] = \preg_replace('#[^a-zA-Z0-9!\-_.*\'() ]#', '-', $uploadEntity->getName());
        }

        return \implode('/', $pathParts) . '.' . $uploadEntity->getExt();
    }

    /**
     * @param int|string $slug
     *
     * @return string
     */
    private function parseSlug($slug): string
    {
        if (\is_int($slug)) {
            $subDir = \floor($slug / 10000) + 1;

            return $subDir . '/' . $slug;
        }

        if (3 > \mb_strlen($slug)) {
            throw new HandlerException('Upload slug must be at least 3 characters long');
        }

        return \mb_substr($slug, 0, 2) . '/' . \mb_substr($slug, 2);
    }
}
