<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\UploadBundle\Component\Handler;

use Cyber\UploadBundle\Component\HandlerException;
use Cyber\UploadBundle\Component\HandlerInterface;
use Cyber\UploadBundle\Component\UploadHolder;

/**
 * A handler for storing files on a local disk.
 */
class LocalStorageHandler implements HandlerInterface
{
    private $storagePath;

    public function __construct(string $storagePath)
    {
        $this->storagePath = $storagePath;
    }

    /**
     * @inheritDoc
     */
    public function upload(UploadHolder $uploadEntity): void
    {
        $file = $uploadEntity->getFile();
        if (!$file) {
            return;
        }
        $realPath = $file->getRealPath();
        if (!$realPath) {
            return;
        }

        $path = $this->getURL($uploadEntity);

        if (!@\copy($realPath, $path)) {
            throw $this->makeException('upload');
        }
    }

    /**
     * @inheritDoc
     */
    public function import(string $fromUrl, UploadHolder $destination): void
    {
        $path = $this->getURL($destination);

        if (!@\copy($fromUrl, $path)) {
            throw $this->makeException('import');
        }
    }

    /**
     * @inheritDoc
     */
    public function delete(UploadHolder $uploadEntity): void
    {
        $path = $this->getURL($uploadEntity);
        if (!@\unlink($path)) {
            throw $this->makeException('delete');
        }
    }

    /**
     * @inheritDoc
     */
    public function copy(UploadHolder $src, UploadHolder $dest): void
    {
        $srcPath = $this->getURL($src);
        $dstPath = $this->getURL($dest);

        if (!@\copy($srcPath, $dstPath)) {
            throw $this->makeException('copy');
        }
    }

    /**
     * @inheritDoc
     */
    public function getURL(UploadHolder $uploadEntity, array $options = []): string
    {
        return \sprintf('%s/%s.%s', $this->storagePath, $uploadEntity->getName(), $uploadEntity->getExt());
    }

    /**
     * @inheritDoc
     */
    public function exists(UploadHolder $uploadEntity): bool
    {
        $path = $this->getURL($uploadEntity);

        return \file_exists($path);
    }

    private function makeException(string $operation): HandlerException
    {
        $error = \error_get_last();

        return new HandlerException(
            \sprintf('Failed to %s file: %s', $operation, ($error ? $error['message'] : 'Unknown error'))
        );
    }
}
