<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\UploadBundle\Component;

use Aws\S3\S3Client;
use Cyber\UploadBundle\Component\Handler\AwsHandler;
use Cyber\UploadBundle\Component\HandlerInterface;
use Cyber\UploadBundle\Component\PathResolver;
use Cyber\UploadBundle\Component\UploadHolder;
use Exception;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Http\Message\UriInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * @internal
 *
 * @coversNothing
 */
class AwsHandlerTest extends WebTestCase
{
    /**
     * @var HandlerInterface
     */
    private $handler;

    /** @var MockObject|S3Client */
    private $client;

    /**
     * @var AwsHandler
     */
    private $handlerWithFileName;

    public function setUp(): void
    {
        $this->client = $this->mockS3Client();
        $resolver     = $this->getMockBuilder(PathResolver::class)
            ->disableOriginalConstructor()
            ->getMock();
        $resolver->expects(static::any())
            ->method('resolve')
            ->willReturn('sub');

        $this->handler             = new AwsHandler($this->client, $resolver, 'crs-sbx', 'uploads/test', 'entity_id');
        $this->handlerWithFileName = new AwsHandler($this->client, $resolver, 'crs-sbx', 'uploads/test', 'file_name');
    }

    public function testUpload(): void
    {
        $this->client->expects(static::exactly(3))
            ->method('upload')
            ->withConsecutive(
                ['crs-sbx', 'uploads/test/sub/1/1.txt'],
                ['crs-sbx', 'uploads/test/sub/ab/c.txt'],
                ['crs-sbx', 'uploads/test/sub/12/3/file name.txt']
            );
        $this->handler->upload($this->mockEntity($this->mockUploadedFile()));
        $this->handler->upload($this->mockEntity($this->mockUploadedFile(), 'abc'));
        $this->handlerWithFileName->upload($this->mockEntity($this->mockUploadedFile(), '123'));
    }

    public function testUploadFileEmpty(): void
    {
        $this->client->expects(static::never())
            ->method('upload');
        $this->handler->upload($this->mockEntity());
    }

    public function testUploadFailure(): void
    {
        $this->expectException('\Cyber\UploadBundle\Component\HandlerException');
        $this->expectExceptionMessage('File upload failed');

        $this->client->expects(static::once())
            ->method('upload')
            ->willThrowException(new Exception('some random exception'));
        $this->handler->upload($this->mockEntity($this->mockUploadedFile()));
    }

    public function testUploadFileNotExist(): void
    {
        $this->expectException(\Cyber\UploadBundle\Component\HandlerException::class);
        $this->expectExceptionMessage('File does not exist at');

        $this->handler->upload($this->mockEntity($this->mockUploadedFile('file_not_exist.txt')));
    }

    public function testDelete(): void
    {
        $this->client->expects(static::once())
            ->method('__call')
            ->with('deleteObject', [['Bucket' => 'crs-sbx', 'Key' => 'uploads/test/sub/1/1.txt']]);
        $this->handler->delete($this->mockEntity($this->mockUploadedFile()));
    }

    public function testCopy(): void
    {
        $this->client->expects(static::once())
            ->method('copy')
            ->with('crs-sbx', 'uploads/test/sub/1/1.txt', 'crs-sbx', 'uploads/test/sub/1/2.txt');

        $this->handler->copy(
            $this->mockEntity($this->mockUploadedFile()),
            $this->mockEntity(null, 2)
        );
    }

    public function testCopyFailure(): void
    {
        $this->expectException(\Cyber\UploadBundle\Component\HandlerException::class);
        $this->expectExceptionMessage('File copy failed.');

        $this->client->expects(static::once())
            ->method('copy')
            ->willThrowException(new Exception('some copy exception'));
        $this->handler->copy(
            $this->mockEntity($this->mockUploadedFile()),
            $this->mockEntity(null, 2)
        );
    }

    public function testGetUrl(): void
    {
        $entity = $this->mockEntity();
        $url    = $this->handler->getURL($entity, ['signed' => true]);
        static::assertEquals('signedUrl', $url);

        $url = $this->handler->getURL($entity);
        static::assertEquals('publicUrl', $url);

        $privateEntity = $this->mockEntity(null, 1, false);
        $url           = $this->handler->getURL($privateEntity, ['signed' => false]);
        // signed = false should be ignore and we get signedUrl
        // TODO assert warning is logged
        static::assertEquals('signedUrl', $url);
    }

    public function testExists(): void
    {
        $this->client->expects(static::once())
            ->method('doesObjectExist')
            ->willReturn(true);

        $entity = $this->mockEntity(null, null);

        $result = $this->handler->exists($entity);
        static::assertFalse($result);

        $entity = $this->mockEntity();

        $result = $this->handler->exists($entity);
        static::assertTrue($result);
    }

    /**
     * @param UploadedFile    $file
     * @param null|int|string $id
     * @param null|bool       $public
     *
     * @return MockObject|UploadHolder
     *
     * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
     */
    private function mockEntity($file = null, $id = 1, $public = true)
    {
        /** @var MockObject|UploadHolder $container */
        $container = $this->getMockBuilder(UploadHolder::class)
            ->getMock();

        $container->expects(static::any())
            ->method('getStorageSlug')
            ->willReturn($id);

        $container->expects(static::any())
            ->method('getExt')
            ->willReturn('txt');

        $container->expects(static::any())
            ->method('getFile')
            ->willReturn($file);

        $container->expects(static::any())
            ->method('getPublic')
            ->willReturn($public);

        $container->expects(static::any())
            ->method('getName')
            ->willReturn('file name');

        return $container;
    }

    /**
     * Mock an UploadedFile Object.
     *
     * @param string $existFile
     *
     * @return MockObject|UploadedFile
     */
    private function mockUploadedFile($existFile = __FILE__)
    {
        /** @var MockObject|UploadedFile $file */
        $file = $this->getMockBuilder(UploadedFile::class)
            ->disableOriginalConstructor()
            ->getMock();

        $file->expects(static::any())
            ->method('getPathName')
            ->willReturn($existFile);

        return $file;
    }

    /**
     * @return MockObject|S3Client
     *
     * @SuppressWarnings(PHPMD.CamelCaseVariableName)
     */
    private function mockS3Client()
    {
        $cmd = $this->getMockBuilder('Aws\\Command')
            ->disableOriginalConstructor()
            ->getMock();

        $request = $this->getMockBuilder('Psr\\Http\\Message\\RequestInterface')
            ->disableOriginalConstructor()
            ->getMock();

        $uriInterface = $this->createMock(UriInterface::class);
        $uriInterface->method('__toString')->willReturn('signedUrl');

        $request->expects(static::any())
            ->method('getUri')
            ->willreturn($uriInterface);

        /** @var MockObject|S3Client $S3Client */
        $S3Client = $this->getMockBuilder('Aws\\S3\\S3Client')
            ->disableOriginalConstructor()
            ->getMock();

        $S3Client->expects(static::any())
            ->method('createPresignedRequest')
            ->willReturn($request);

        $S3Client->expects(static::any())
            ->method('getCommand')
            ->willReturn($cmd);

        $S3Client->expects(static::any())
            ->method('getObjectUrl')
            ->willReturn('publicUrl');

        return $S3Client;
    }
}
