<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\UploadBundle;

use Cyber\UploadBundle\Component\Handler\AwsHandler;
use Cyber\UploadBundle\Component\Handler\LocalStorageHandler;
use Cyber\UploadBundle\Component\Handler\NoOpHandler;
use LogicException;
use RuntimeException;
use Symfony\Component\Config\Definition\Configurator\DefinitionConfigurator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Loader\Configurator\ContainerConfigurator;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\Bundle\AbstractBundle;

/**
 * @phpstan-type MappingShape array<int,array{class: string, path: string}>
 * @phpstan-type LocalShape array{handler: 'local', local: array{path: string, web_path: string}, mapping: MappingShape}
 * @phpstan-type ServiceShape array{handler: 'service', service: string, mapping: MappingShape}
 * @phpstan-type NoHandlerShape array{mapping: MappingShape}
 */
class CyberUploadBundle extends AbstractBundle
{
    public function configure(DefinitionConfigurator $definition): void
    {
        // @formatter:off
        $definition->rootNode()
            ->children()
                ->enumNode('handler')
                    ->values(['aws', 'service', 'local', 'noop'])
                    ->cannotBeEmpty()
                ->end()
                ->arrayNode('aws')
                    ->info('when handler is \'aws\' these parameters are required, none should be empty')
                    ->children()
                        ->scalarNode('bucket')
                            ->info('the bucket where uploads will be stored')
                        ->end()
                        ->scalarNode('path')
                            ->info('the path inside the bucket where uploads will be stored')
                            ->example('company/uploads')
                        ->end()
                        ->scalarNode('key')
                            ->info('the key used for authentication with S3')
                            ->isRequired()
                            ->cannotBeEmpty()
                        ->end()
                        ->scalarNode('secret')
                            ->info('the secret used for authentication with S3')
                            ->isRequired()
                            ->cannotBeEmpty()
                        ->end()
                    ->end()
                ->end()//aws
                ->arrayNode('local')
                    ->info('when handler is "local" these parameters are required')
                    ->children()
                        ->scalarNode('path')
                            ->info('Absolute path to store the files in')
                            ->cannotBeEmpty()
                        ->end()
                        ->scalarNode('web_path')
                            ->info('Web server url path where the above storage path is mounted')
                            ->cannotBeEmpty()
                        ->end()
                    ->end()
                ->end()//local
                ->enumNode('naming_strategy')
                    ->values(['entity_id', 'file_name'])
                    ->defaultValue('file_name')
                ->end() // naming_strategy
                ->scalarNode('service')
                    ->info('When handler is "service" the service id should be specified here')
                ->end()
                ->arrayNode('mapping')
                    ->prototype('array')
                        ->children()
                            ->scalarNode('class')
                                ->info('Fully qualified class name to map to a path below')
                                ->cannotBeEmpty()
                            ->end()
                            ->scalarNode('path')
                                ->info('The storage path the above class is mapped to')
                                ->cannotBeEmpty()
                            ->end()
                        ->end()
                    ->end()
                ->end() //mapping
            ->end();
        // @formatter:on
    }

    /**
     * @param LocalShape|NoHandlerShape|ServiceShape $config
     */
    public function loadExtension(array $config, ContainerConfigurator $container, ContainerBuilder $builder): void
    {
        if (!isset($config['handler'])) {
            // if we don't have required config, do not load anything.
            // user probably added the bundle but was not ready to configure it yet.
            return;
        }

        $container->import('../config/services.php');
        $this->setupHandlerService($config, $builder, $container);

        if ($config['mapping']) {
            $mapping = [];

            foreach ($config['mapping'] as $key => $map) {
                $class      = $map['class'];
                $path       = \trim(\str_replace('\\', '/', $map['path']), '/');
                $configPath = 'cyber_upload.mapping.' . $key;

                if (\array_key_exists($class, $mapping)) {
                    $ex = new LogicException('Duplicate entry exists for class: ' . $class . ' at ' . $configPath);
                    throw $ex;
                }

                if (\in_array($path, $mapping, true)) {
                    $ex = new LogicException('Duplicate entry exists for path: ' . $path . ' at ' . $configPath);
                    throw $ex;
                }

                $mapping[$class] = $path;
            }
            $container->parameters()
                ->set('cyber.upload.mapping', $mapping);
        }
    }

    /**
     * @param LocalShape|ServiceShape $config
     */
    private function setupHandlerService(array $config, ContainerBuilder $container, ContainerConfigurator $configurator): void
    {
        // for service just setup alias
        if ('service' === $config['handler']) {
            $container->setAlias('cyber_upload.file.handler', $config['service']);

            return;
        }

        // otherwise do more configurations
        switch ($config['handler']) {
            case 'aws':
                $configurator->import('../config/aws-handler.php');
                $class = AwsHandler::class;
                $arg   = [
                    '$s3Client'       => new Reference('cyber.upload.s3client'),
                    '$resolver'       => new Reference('cyber_upload.path.resolver'),
                    '$bucket'         => $config['aws']['bucket'],
                    '$path'           => $config['aws']['path'],
                    '$namingStrategy' => $config['naming_strategy'],
                ];
                $container->setParameter('cyber.upload.key', $config['aws']['key']);
                $container->setParameter('cyber.upload.secret', $config['aws']['secret']);
                break;
            case 'local':
                $class = LocalStorageHandler::class;
                $arg   = [
                    '$storagePath' => $config['local']['path'],
                    '$webPath'     => $config['local']['web_path'],
                ];
                break;
            case 'noop':
                $class = NoOpHandler::class;
                $arg   = [];
                break;
            default:
                throw new RuntimeException('Unsupported handler specified');
        }

        $container->setDefinition(
            'cyber_upload.file.handler',
            (new Definition($class))
                ->setArguments($arg)
                ->setLazy(true) //need to be lazy cause we inject into twig extension, only helps if lazy is supported
        );
    }
}
